//@author: fibo
//@description: multilight shader using Attractor3d.fxh vertex transform
//@tags: attractor
//@credits: joreg for the PhongMultiPoint.fx
//shading:         phong
//lighting model:  blinn
//lighting type:   point

// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;         //view matrix as set via Renderer (DX9)
float4x4 tWV: WORLDVIEW;
float4x4 tWVP: WORLDVIEWPROJECTION;
float4x4 tP: PROJECTION;   //projection matrix as set via Renderer (DX9)

//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
// EDIT HERE TO SPREAD THE ATTRACTOR
#define COUNT 2
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

float3 lPos[COUNT] <string uiname="Light Position";>;       //light position in world space
float4 lAmb[COUNT]: COLOR <String uiname="Ambient Color";>;
float4 lDiff[COUNT]: COLOR <String uiname="Diffuse Color";>;
float4 lSpec[COUNT]: COLOR <String uiname="Specular Color";>;
float lPower[COUNT] <String uiname="Power"; float uimin=0.0;>;     //shininess of specular highlight
float lRange[COUNT] <String uiname="Light Range"; float uimin=0.0;>;
float lGamma[COUNT] <String uiname="Range Gamma"; float uimin=0.0;>;

float3 AttractorCenter[COUNT];
float AttractorRadius[COUNT];
float AttractorPower[COUNT];
float AttractorStrength[COUNT];

#include "Attractor3d.fxh"

//texture
texture Tex <string uiname="Texture";>;
sampler Samp = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (Tex);          //apply a texture to the sampler
    MipFilter = LINEAR;         //set the sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

float4x4 tTex: TEXTUREMATRIX <string uiname="Texture Transform";>;

struct vs2ps
{
    float4 PosWVP: POSITION;
    float4 TexCd : TEXCOORD0;
    float3 NormV: COLOR0;       //using any free semantics here
    float3 ViewDirV: COLOR1;
    float3 PosW: TEXCOORD1;
    float3 LightDirV[COUNT]: TEXCOORD2;
};

// --------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// --------------------------------------------------------------------------------------------------

vs2ps VS(
    float4 PosO: POSITION,
    float3 NormO: NORMAL,
    float4 TexCd : TEXCOORD0)
{
    //inititalize all fields of output struct with 0
    vs2ps Out = (vs2ps)0;

    Out.PosW = mul(PosO, tW);

    PosO = mul(PosO, tW);

    for (int i=0; i<COUNT; i++) {
        PosO.xyz=attractor3d(PosO.xyz,AttractorCenter[i],AttractorPower[i],AttractorStrength[i],AttractorRadius[i]);
    }
          
    PosO = mul(PosO, tV);
                  
    Out.PosWVP = mul(PosO, tP);
    
    //inverse light direction in view space
    for (int i=0; i<COUNT; i++)
    {
        float3 lightDirW = normalize(lPos[i] - Out.PosW);
        Out.LightDirV[i] = mul(lightDirW, tV);
    }
    
    //normal in view space
    Out.NormV = normalize(mul(NormO, tWV));

    //position (projected)
    //Out.PosWVP  = mul(PosO, tWVP);
    Out.TexCd = mul(TexCd, tTex);
    Out.ViewDirV = -normalize(mul(PosO, tWV));
    return Out;
}

// --------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// --------------------------------------------------------------------------------------------------



float4 ComputeBlinn(int LightIndex, vs2ps In)
{
    float d = distance(In.PosW, lPos[LightIndex]);
    d = max(lRange[LightIndex]-d, 0);
    d = pow(d, lGamma[LightIndex]);

    float4 amb = lAmb[LightIndex] * d;
    amb.a = 1;

    //halfvector
    float3 H = normalize(In.ViewDirV + In.LightDirV[LightIndex]);

    //compute blinn lighting
    float4 shades = lit(dot(In.NormV, In.LightDirV[LightIndex]), dot(In.NormV, H), lPower[LightIndex]);

    float4 diff = lDiff[LightIndex] * shades.y * d;
    diff.a = 1;
    //reflection vector (view space)
    float3 R = normalize(2 * dot(In.NormV, In.LightDirV[LightIndex]) * In.NormV - In.LightDirV[LightIndex]);
    //normalized view direction (view space)
    float3 V = normalize(In.ViewDirV);

    //calculate specular light
    float4 spec = pow(max(dot(R, V),0), lPower[LightIndex]*.2) * lSpec[LightIndex];
    
    return amb + diff + spec;
}

float4 PS(vs2ps In): COLOR 
{
    float4 col = 0;
    for (int i=0; i<COUNT; i++)
    {
      col += ComputeBlinn(i, In);
    }
  
    col.a = 1;

    float4 tex = tex2D(Samp, In.TexCd);

    col.rgb *= tex;
    return col;
}


// --------------------------------------------------------------------------------------------------
// TECHNIQUES:
// --------------------------------------------------------------------------------------------------

technique TPhongPoint 
{
    pass P0 
    {
       VertexShader = compile vs_2_0 VS();
       PixelShader = compile ps_3_0 PS();
    }
}
