//@author: misak
//@credits: ATI
//----------------------------------------------------------------------------------//
// Phenomenological car paint visual effect: Simulates build up of paint coats     
// on the metallic surface of a car or any other object. This visual effect        
// emulates the suspended layers of microflakes in the paint coat as well as        
// blends between several colors of paint based on the viewing angle.          
//                                        
// Author: Natalya Tatarchuk                
//      (based on the original assembly shader written by John Isidoro)       
//                                        
// Used in ATI's Car demo for Radeon 9700 launch, found here:             
//       http://www.ati.com/developer/demos/r9700.html                
//                                        
// (C) ATI Research, 2003                   
//----------------------------------------------------------------------------------//


// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD ;        //the models world matrix
float4x4 tV: VIEW ;         //view matrix as set via Renderer (EX9)
float4x4 tP: PROJECTION ;
float4x4 tWVP: WORLDVIEWPROJECTION ;
float4x4 tVIT : VIEWINVERSETRANSPOSE ; 

float4x4 tVP : VIEWPROJECTION ;


texture normalMap <string uiname="NormalMap";>;
sampler normalMapSamp = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (normalMap);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
	MIPMAPLODBIAS = 0.000000;
};


texture microflakeNMap <string uiname="microflakeNMap";>;
sampler microflakeNMapSamp = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (microflakeNMap);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
	MIPMAPLODBIAS = 0.000000;
};


texture showroomMap <string uiname="showroomMap";>;
samplerCUBE showroomMapSamp = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (showroomMap);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
	MIPMAPLODBIAS = 0.000000;
};


//texture transformation marked with semantic TEXTUREMATRIX to achieve symmetric transformations
float4x4 tTex: TEXTUREMATRIX <string uiname="Texture Transform";>;


float4 paintColorMid  : COLOR <String uiname="paintColorMid";>  = {0.6, 0.0, 0.0, 0.0};
float4 paintColor2 : COLOR <String uiname="paintColor2";>  = {0.0, 0.35, -0.35, 0.0 };
float4 paintColor0 : COLOR <String uiname="paintColor0";>  = {0.4, 0.0, 0.349, 0.0};
float4 flakeLayerColor : COLOR <String uiname="flakeLayerColor";>  = {0.5, 0.5, 0.0, 0.0};


float normalPerturbation = {1.0};
float microflakePerturbationA = {0.1};
float microflakePerturbation = {1.0} ;
float brightnessFactor = {8.0};
float glossLevel = {0.0};


//the data structure: "vertexshader to pixelshader"
//used as output data with the VS function
//and as input data with the PS function
struct vs2ps
{
   float4 Pos        : POSITION;
   float2 TexCd      : TEXCOORD0;
   float3 Tangent    : TEXCOORD1;
   float3 Binormal   : TEXCOORD2;
   float3 Normal     : TEXCOORD3;
   float3 View       : TEXCOORD4;
   float3 SparkleTex : TEXCOORD5;
	
};

// --------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// --------------------------------------------------------------------------------------------------
vs2ps VS(
  		 float4 PosO     : POSITION, 
         float3 Normal   : NORMAL,
         float2 TexCd    : TEXCOORD0,
         float3 Tangent  : TANGENT,
         float3 Binormal : BINORMAL)
{
    //declare output struct
    vs2ps Out;

    //transform position
    Out.Pos = mul(PosO, tWVP);
    
    //transform texturecoordinates
///	Out.TexCd  =  mul(TexCd, tTex); 
	Out.TexCd = mul(float4(TexCd, 0, 1), tTex);
	

	
   // Compute view vector: 
   Out.View = normalize( mul(tVIT, float4( 0, 0, 0, 1)) - PosO );

   // Propagate texture coordinates:
   //инвертируем текстуру по оси  y	
    Out.TexCd.y  = 1 - Out.TexCd.y ; 
	
	
   // Propagate tangent, binormal, and normal vectors to pixel shader:
   Out.Normal   = Normal;
   Out.Tangent  = Tangent;
   Out.Binormal = -Binormal;
  
	
   // Compute microflake tiling factor:
   Out.SparkleTex = float4( Out.TexCd * 20.0, 0, 1 );
   
   return Out;
	

}

// --------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// --------------------------------------------------------------------------------------------------

float4 main( float4 Diff        : COLOR0,
          float2 Tex        : TEXCOORD0,
          float3 Tangent    : TEXCOORD1,
          float3 Binormal   : TEXCOORD2,
          float3 Normal     : TEXCOORD3,
          float3 View       : TEXCOORD4,
          float3 SparkleTex : TEXCOORD5 ) : COLOR
{ 
   // fetch from the incoming normal map:
   float3 vNormal = tex2D(normalMapSamp, Tex );

////	vNormal.rg =  - vNormal.rg ;
   // Scale and bias fetched normal to move into [-1.0, 1.0] range:
   vNormal = (2 * vNormal - 1.0) * 1;

   // Microflakes normal map is a high frequency normalized
      // vector noise map which is repeated across all surface. 
   // Fetching the value from it for each pixel allows us to 
   // compute perturbed normal for the surface to simulate
   // appearance of microflakes suspected in the coat of paint:
   float3 vFlakesNormal = tex2D( microflakeNMapSamp, SparkleTex );
   
   // Don't forget to bias and scale to shift color into [-1.0, 1.0] range:
   vFlakesNormal = 2 * vFlakesNormal - 1.0;

   // This shader simulates two layers of microflakes suspended in 
   // the coat of paint. To compute the surface normal for the first layer,
   // the following formula is used: 
   //   Np1 = ( a * Np + b * N ) /  || a * Np + b * N || where a << b
   //
   float3 vNp1 = microflakePerturbationA * vFlakesNormal + normalPerturbation * vNormal ; 

   // To compute the surface normal for the second layer of microflakes, which
   // is shifted with respect to the first layer of microflakes, we use this formula:
   //    Np2 = ( c * Np + d * N ) / || c * Np + d * N || where c == d
   // 
   float3 vNp2 = microflakePerturbation * ( vFlakesNormal + vNormal ) ;

   // The view vector (which is currently in world space) needs to be normalized.
   // This vector is normalized in the pixel shader to ensure higher precision of
   // the resultinv view vector. For this highly detailed visual effect normalizing
   // the view vector in the vertex shader and simply interpolating it is insufficient
   // and produces artifacts.
   float3 vView =  normalize( View );

   // Transform the surface normal into world space (in order to compute reflection
   // vector to perform environment map look-up):
   float3x3 mTangentToWorld = transpose( float3x3( Tangent, Binormal, Normal ) );
   float3   vNormalWorld    = normalize( mul( mTangentToWorld, vNormal ));

   // Compute reflection vector resulted from the clear coat of paint on the metallic
   // surface:
   float  fNdotV     = saturate(dot( vNormalWorld, vView));
   float3 vReflection = 2 * vNormalWorld * fNdotV - vView;

   // Here we just use a constant gloss value to bias reading from the environment
   // map, however, in the real demo we use a gloss map which specifies which 
   // regions will have reflection slightly blurred.
   float fEnvBias = glossLevel;

   // Sample environment map using this reflection vector:
   float4 envMap = texCUBEbias( showroomMapSamp, float4( vReflection, fEnvBias ) );

   // Premultiply by alpha:
      envMap.rgb = envMap.rgb * envMap.a;

   // Brighten the environment map sampling result:
   envMap.rgb *= brightnessFactor;

   // Compute modified Fresnel term for reflections from the first layer of
   // microflakes. First transform perturbed surface normal for that layer into 
   // world space and then compute dot product of that normal with the view vector:
   float3 vNp1World = normalize( mul( mTangentToWorld, vNp1) );
   float  fFresnel1 = saturate( dot( vNp1World, vView ));

   // Compute modified Fresnel term for reflections from the second layer of 
   // microflakes. Again, transform perturbed surface normal for that layer into 
   // world space and then compute dot product of that normal with the view vector:
   float3 vNp2World = normalize( mul( mTangentToWorld, vNp2 ));
   float  fFresnel2 = saturate( dot( vNp2World, vView ));   

   //
   // Compute final paint color: combines all layers of paint as well as two layers
   // of microflakes
   //
   float  fFresnel1Sq = fFresnel1 * fFresnel1;

   float4 paintColor = fFresnel1   * paintColor0 + 
              fFresnel1Sq * paintColorMid +
              fFresnel1Sq * fFresnel1Sq * paintColor2 +
              pow( fFresnel2, 16 ) * flakeLayerColor;

   // Combine result of environment map reflection with the paint color:
   float  fEnvContribution = 1.0 - 0.5 * fNdotV;

   float4 finalColor;
   finalColor.a = 1.0;
   finalColor.rgb = envMap * fEnvContribution + paintColor;
              
   return finalColor;
	
}



// --------------------------------------------------------------------------------------------------
// TECHNIQUES:
// --------------------------------------------------------------------------------------------------

technique TMultiToneCarPaint
{
    pass P0
    {
        //Wrap0 = U;  // useful when mesh is round like a sphere
        VertexShader = compile vs_3_0 VS();
        PixelShader  = compile ps_3_0 main();
    }
}