#region usings
using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel.Composition;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;

using VVVV.Core.Logging;
#endregion usings

namespace VVVV.Nodes
{
	#region PluginInfo
	[PluginInfo(Name = "PDollar", Category = "Value", Help = "Basic template with one value in/out", Tags = "")]
	#endregion PluginInfo
	public class ValuePDollarNode : IPluginEvaluate
	{
		#region fields & pins
		[Input("X", DefaultValue = 1.0)]
		ISpread<float> FXInput;
		
		[Input("Y", DefaultValue = 1.0)]
		ISpread<float> FYInput;
		
		[Input("Strokes Count", DefaultValue = 1.0)]
		ISpread<int> FStrokeIndex;
		
		[Input("Recognize", IsSingle = true)]
		ISpread<bool> FRecognize;
		
		[Input("Template X", DefaultValue = 1.0)]
		ISpread<float> FXTemplate;
		
		[Input("Template Y", DefaultValue = 1.0)]
		ISpread<float> FYTemplate;
		
		[Input("Template Stroke Index", DefaultValue = 1.0)]
		ISpread<int> FIndexTemplate;
		
		[Input("Gesture Template Names")]
		ISpread<string> FNameTemplate;
		
		[Input("Load Training Data", IsSingle = true)]
		ISpread<bool> FLoadGestures;
		
		[Input("Output Candidate", IsSingle = true)]
		ISpread<bool> FShowCandidate;

		[Output("Recognized Gesture")]
		ISpread<string> FOutput;
		
		[Output("Gesture output for file save")]
		ISpread<string> FGestureOutput;

		[Import()]
		ILogger FLogger;
		#endregion fields & pins
		
		Gesture[] trainingSet = null;

		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			FOutput.SliceCount = 1;

			if(FRecognize[0])
			{
				if(trainingSet != null && FStrokeIndex.SliceCount > 0)
				{
					Gesture candidate = ReadInput();
					string result = PointCloudRecognizer.Classify(candidate, trainingSet);
					FOutput[0] = result;	
				}else{
					FOutput[0] = "No training data loaded!";
				}
			}
			
			if(FShowCandidate[0] && FStrokeIndex.SliceCount > 0)
			{
				FGestureOutput.SliceCount = 1;
				Gesture candidate = ReadInput();
				FGestureOutput[0] = WriteGesture(candidate.Points);
			}
			
			if(FLoadGestures[0])
			{
				trainingSet = LoadTrainingSet();
			}
		}
		
		/// <summary>
        /// Loads training gesture samples from XML files
        /// </summary>
        /// <returns></returns>
        private Gesture ReadInput()
        {
        	List<Point> points = new List<Point>();
        	int pointOffset = 0;
        	for(int x=0; x<FStrokeIndex.SliceCount; x++)
        	{
        		for(int y=0; y<FStrokeIndex[x]; y++)
        		{
        			points.Add(new Point(FXInput[y + pointOffset], FYInput[y + pointOffset], x+1));
        		}
        		pointOffset += FStrokeIndex[x];
        	}
        	
        	return new Gesture( points.ToArray() );       	
        }
		
		/// <summary>
        /// Loads training gesture samples from XML files
        /// </summary>
        /// <returns></returns>
        private Gesture[] LoadTrainingSet()
        {
            List<Gesture> gestures = new List<Gesture>();
            List<Point> tpoints = new List<Point>();
        	int gestureOffset = 0;
        	for(int g=0; g<FNameTemplate.SliceCount; g++)
        	{
        		//read in next 64 values
	        	for(int p=0; p<64; p++)
        		{
        			tpoints.Add( new Point(FXTemplate[p+gestureOffset], 
        						FYTemplate[p+gestureOffset], 
        						FIndexTemplate[p+gestureOffset]) );
        		}
        		gestureOffset += 64;
        		gestures.Add( new Gesture(tpoints.ToArray(), FNameTemplate[g]) );
        		tpoints.Clear();
        	}
        	
            return gestures.ToArray();
        }
		
		private string WriteGesture(Point[] points)
		{
			StringWriter sw = new StringWriter();
			int currentStroke = -1;
            for (int i = 0; i < points.Length; i++)
            {
                if (points[i].StrokeID != currentStroke)
                {
                    if (i > 0)
                        sw.WriteLine("\t</Stroke>");
                    currentStroke = points[i].StrokeID;
                    sw.WriteLine("\t<Stroke index=\"{0}\">", currentStroke);
                }

                sw.WriteLine("\t\t<Point X=\"{0}\" Y=\"{1}\" index=\"{2}\" />",
                    points[i].X, points[i].Y, currentStroke
                );
            }
            sw.WriteLine("\t</Stroke>");
			
			return sw.ToString();
		}
	}
}

/**
 * The $P Point-Cloud Recognizer (.NET Framework 4.0 C# version)
 *
 * 	    Radu-Daniel Vatavu, Ph.D.
 *	    University Stefan cel Mare of Suceava
 *	    Suceava 720229, Romania
 *	    vatavu@eed.usv.ro
 *
 *	    Lisa Anthony, Ph.D.
 *      UMBC
 *      Information Systems Department
 *      1000 Hilltop Circle
 *      Baltimore, MD 21250
 *      lanthony@umbc.edu
 *
 *	    Jacob O. Wobbrock, Ph.D.
 * 	    The Information School
 *	    University of Washington
 *	    Seattle, WA 98195-2840
 *	    wobbrock@uw.edu
 *
 * The academic publication for the $P recognizer, and what should be 
 * used to cite it, is:
 *
 *	Vatavu, R.-D., Anthony, L. and Wobbrock, J.O. (2012).  
 *	  Gestures as point clouds: A $P recognizer for user interface 
 *	  prototypes. Proceedings of the ACM Int'l Conference on  
 *	  Multimodal Interfaces (ICMI '12). Santa Monica, California  
 *	  (October 22-26, 2012). New York: ACM Press, pp. 273-280.
 *
 * This software is distributed under the "New BSD License" agreement:
 *
 * Copyright (c) 2012, Radu-Daniel Vatavu, Lisa Anthony, and 
 * Jacob O. Wobbrock. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *    * Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    * Neither the names of the University Stefan cel Mare of Suceava, 
 *	    University of Washington, nor UMBC, nor the names of its contributors 
 *	    may be used to endorse or promote products derived from this software 
 *	    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL Radu-Daniel Vatavu OR Lisa Anthony
 * OR Jacob O. Wobbrock BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
**/

public class PointCloudRecognizer
{
    /// <summary>
    /// Main function of the $P recognizer.
    /// Classifies a candidate gesture against a set of training samples.
    /// Returns the class of the closest neighbor in the training set.
    /// </summary>
    /// <param name="candidate"></param>
    /// <param name="trainingSet"></param>
    /// <returns></returns>
    public static string Classify(Gesture candidate, Gesture[] trainingSet)
    {
        float minDistance = float.MaxValue;
        string gestureClass = "";
        foreach (Gesture template in trainingSet)
        {
            float dist = GreedyCloudMatch(candidate.Points, template.Points);
            if (dist < minDistance)
            {
                minDistance = dist;
                gestureClass = template.Name + ";" + dist.ToString();
            }
        }
        return gestureClass;
    }

    /// <summary>
    /// Implements greedy search for a minimum-distance matching between two point clouds
    /// </summary>
    /// <param name="points1"></param>
    /// <param name="points2"></param>
    /// <returns></returns>
    private static float GreedyCloudMatch(Point[] points1, Point[] points2)
    {
        int n = points1.Length; // the two clouds should have the same number of points by now
        float eps = 0.5f;       // controls the number of greedy search trials (eps is in [0..1])
        int step = (int)Math.Floor(Math.Pow(n, 1.0f - eps));
        float minDistance = float.MaxValue;
        for (int i = 0; i < n; i += step)
        {
            float dist1 = CloudDistance(points1, points2, i);   // match points1 --> points2 starting with index point i
            float dist2 = CloudDistance(points2, points1, i);   // match points2 --> points1 starting with index point i
            minDistance = Math.Min(minDistance, Math.Min(dist1, dist2));
        }
        return minDistance;
    }

    /// <summary>
    /// Computes the distance between two point clouds by performing a minimum-distance greedy matching
    /// starting with point startIndex
    /// </summary>
    /// <param name="points1"></param>
    /// <param name="points2"></param>
    /// <param name="startIndex"></param>
    /// <returns></returns>
    private static float CloudDistance(Point[] points1, Point[] points2, int startIndex)
    {
        int n = points1.Length;       // the two clouds should have the same number of points by now
        bool[] matched = new bool[n]; // matched[i] signals whether point i from the 2nd cloud has been already matched
        Array.Clear(matched, 0, n);   // no points are matched at the beginning

        float sum = 0;  // computes the sum of distances between matched points (i.e., the distance between the two clouds)
        int i = startIndex;
        do
        {
            int index = -1;
            float minDistance = float.MaxValue;
            for(int j = 0; j < n; j++)
                if (!matched[j])
                {
                    float dist = Geometry.SqrEuclideanDistance(points1[i], points2[j]);  // use squared Euclidean distance to save some processing time
                    if (dist < minDistance)
                    {
                        minDistance = dist;
                        index = j;
                    }
                }
            matched[index] = true; // point index from the 2nd cloud is matched to point i from the 1st cloud
            float weight = 1.0f - ((i - startIndex + n) % n) / (1.0f * n);
            sum += weight * minDistance; // weight each distance with a confidence coefficient that decreases from 1 to 0
            i = (i + 1) % n;
        } while (i != startIndex);
        return sum;
    }
}
/// <summary>
/// Implements a 2D Point that exposes X, Y, and StrokeID properties.
/// StrokeID is the stroke index the point belongs to (e.g., 0, 1, 2, ...) that is filled by counting pen down/up events.
/// </summary>
public class Point
{
    public float X, Y;
    public int StrokeID;      

    public Point(float x, float y, int strokeId)
    {
        this.X = x;
        this.Y = y;
        this.StrokeID = strokeId;
    }
}

/// <summary>
/// Implements a gesture as a cloud of points (i.e., an unordered set of points).
/// Gestures are normalized with respect to scale, translated to origin, and resampled into a fixed number of 32 points.
/// </summary>
public class Gesture
{
    public Point[] Points = null;            // gesture points (normalized)
    public string Name = "";                 // gesture class
    private const int SAMPLING_RESOLUTION = 64;

    /// <summary>
    /// Constructs a gesture from an array of points
    /// </summary>
    /// <param name="points"></param>
    public Gesture(Point[] points, string gestureName = "")
    {
        this.Name = gestureName;
        
        // normalizes the array of points with respect to scale, origin, and number of points
        this.Points = Scale(points);
        this.Points = TranslateTo(Points, Centroid(Points));
        this.Points = Resample(Points, SAMPLING_RESOLUTION);
    }

    #region gesture pre-processing steps: scale normalization, translation to origin, and resampling

    /// <summary>
    /// Performs scale normalization with shape preservation into [0..1]x[0..1]
    /// </summary>
    /// <param name="points"></param>
    /// <returns></returns>
    private Point[] Scale(Point[] points)
    {
        float minx = float.MaxValue, miny = float.MaxValue, maxx = float.MinValue, maxy = float.MinValue;
        for (int i = 0; i < points.Length; i++)
        {
            if (minx > points[i].X) minx = points[i].X;
            if (miny > points[i].Y) miny = points[i].Y;
            if (maxx < points[i].X) maxx = points[i].X;
            if (maxy < points[i].Y) maxy = points[i].Y;
        }

        Point[] newPoints = new Point[points.Length];
        float scale = Math.Max(maxx - minx, maxy - miny);
        for (int i = 0; i < points.Length; i++)
            newPoints[i] = new Point((points[i].X - minx) / scale, (points[i].Y - miny) / scale, points[i].StrokeID);
        return newPoints;
    }

    /// <summary>
    /// Translates the array of points by p
    /// </summary>
    /// <param name="points"></param>
    /// <param name="p"></param>
    /// <returns></returns>
    private Point[] TranslateTo(Point[] points, Point p)
    {
        Point[] newPoints = new Point[points.Length];
        for (int i = 0; i < points.Length; i++)
            newPoints[i] = new Point(points[i].X - p.X, points[i].Y - p.Y, points[i].StrokeID);
        return newPoints;
    }

    /// <summary>
    /// Computes the centroid for an array of points
    /// </summary>
    /// <param name="points"></param>
    /// <returns></returns>
    private Point Centroid(Point[] points)
    {
        float cx = 0, cy = 0;
        for (int i = 0; i < points.Length; i++)
        {
            cx += points[i].X;
            cy += points[i].Y;
        }
        return new Point(cx / points.Length, cy / points.Length, 0);
    }

    /// <summary>
    /// Resamples the array of points into n equally-distanced points
    /// </summary>
    /// <param name="points"></param>
    /// <param name="n"></param>
    /// <returns></returns>
    public Point[] Resample(Point[] points, int n)
    {
        Point[] newPoints = new Point[n];
        newPoints[0] = new Point(points[0].X, points[0].Y, points[0].StrokeID);
        int numPoints = 1;

        float I = PathLength(points) / (n - 1); // computes interval length
        float D = 0;
        for (int i = 1; i < points.Length; i++)
        {
            if (points[i].StrokeID == points[i - 1].StrokeID)
            {
                float d = Geometry.EuclideanDistance(points[i - 1], points[i]);
                if (D + d >= I)
                {
                    Point firstPoint = points[i - 1];
                    while (D + d >= I)
                    {
                        // add interpolated point
                        float t = Math.Min(Math.Max((I - D) / d, 0.0f), 1.0f);
                        if (float.IsNaN(t)) t = 0.5f;
                        newPoints[numPoints++] = new Point(
                            (1.0f - t) * firstPoint.X + t * points[i].X,
                            (1.0f - t) * firstPoint.Y + t * points[i].Y,
                            points[i].StrokeID
                        );

                        // update partial length
                        d = D + d - I;
                        D = 0;
                        firstPoint = newPoints[numPoints - 1];
                    }
                    D = d;
                }
                else D += d;
            }
        }

        if (numPoints == n - 1) // sometimes we fall a rounding-error short of adding the last point, so add it if so
            newPoints[numPoints++] = new Point(points[points.Length - 1].X, points[points.Length - 1].Y, points[points.Length - 1].StrokeID);
        return newPoints;
    }

    /// <summary>
    /// Computes the path length for an array of points
    /// </summary>
    /// <param name="points"></param>
    /// <returns></returns>
    private float PathLength(Point[] points)
    {
        float length = 0;
        for (int i = 1; i < points.Length; i++)
            if (points[i].StrokeID == points[i - 1].StrokeID)
                length += Geometry.EuclideanDistance(points[i - 1], points[i]);
        return length;
    }

    #endregion
}

public class Geometry
{
    /// <summary>
    /// Computes the Squared Euclidean Distance between two points in 2D
    /// </summary>
    public static float SqrEuclideanDistance(Point a, Point b)
    {
        return (a.X - b.X) * (a.X - b.X) + (a.Y - b.Y) * (a.Y - b.Y);
    }

    /// <summary>
    /// Computes the Euclidean Distance between two points in 2D
    /// </summary>
    public static float EuclideanDistance(Point a, Point b)
    {
        return (float)Math.Sqrt(SqrEuclideanDistance(a, b));
    }
}