// Levels.fx
//
// a "levels" oriented pixel shader
// http://pippin.gimp.org/image_processing/chap_point.html has a bunch of the
// gimps algorithms for this.
//
// @version : 1.0
// @author : mtAllen@gmail.com
// @date 25.02.2006

// -----------------------------------------------------------------------------
// PARAMETERS:
// -----------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;         //view matrix as set via Renderer (EX9)
float4x4 tP: PROJECTION;
float4x4 tWVP: WORLDVIEWPROJECTION;

//texture
texture Image <string uiname="Input Image";>;
sampler Samp = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (Image);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};


// texture transformation marked with semantic TEXTUREMATRIX to achieve
// symmetric transformations
float4x4 tTex: TEXTUREMATRIX <string uiname="Texture Transform";>;

//
// inputs
//

float blackPoint
<
    string uiname="BlackPoint";
    float uimin=0.0;
    float uimax=1.0;
> = 0.0;

float whitePoint
<
    string uiname="WhitePoint";
    float uimin=0.0;
    float uimax=1.0;
> = 1.0;

float grayPoint
<
    string uiname="GrayPoint";
    float uimin=0.01;
    float uimax=10.0;
> = 0.75;

float blackOffset
<
    string uiname="BlackOffset";
    float uimin=0.0;
    float uimax=1.0;
> = 0.0;

float whiteOffset
<
    string uiname="WhiteOffset";
    float uimin=0.0;
    float uimax=1.0;
> = 0.0;


float rOffset  <string uiname="Red";>;
float gOffset  <string uiname="Green";>;
float bOffset  <string uiname="Blue";>;


//the data structure: "vertexshader to pixelshader"
//used as output data with the VS function
//and as input data with the PS function
struct vs2ps
{
    float4 Pos  : POSITION;
    float2 TexCd : TEXCOORD0;
};

// -----------------------------------------------------------------------------
// VERTEXSHADERS
// -----------------------------------------------------------------------------
vs2ps VS(
    float4 PosO  : POSITION,
    float4 TexCd : TEXCOORD0)
{
    //inititalize all fields of output struct with 0
    vs2ps Out = (vs2ps)0;

    //transform position
    Out.Pos = mul(PosO, tWVP);

    //transform texturecoordinates
    Out.TexCd = mul(TexCd, tTex);

    return Out;
}

// -----------------------------------------------------------------------------
// PIXELSHADERS:
// -----------------------------------------------------------------------------

// levels function
float levels(float value)
{
    // clamp the incoming variables
    blackPoint = clamp(blackPoint, blackPoint, whitePoint - 0.008);
    whitePoint = clamp(whitePoint, blackPoint + 0.008, whitePoint);
    grayPoint = clamp(grayPoint, blackPoint + 0.004, whitePoint - 0.004);

    // normalize
    value = saturate((value - blackPoint) / ( whitePoint - blackPoint));
    //do the mid point
    value = pow(value, grayPoint);
    // get the value back in range
    value = value * ( whiteOffset - blackOffset) + blackOffset;

    return value;
}
// Technique : Levels
// main pixel shader
float4 psLevels(vs2ps In): COLOR
{
    float4 col = tex2D(Samp, In.TexCd);

   col.r = levels(col.r + rOffset);
   col.g = levels(col.g + gOffset);
   col.b = levels(col.b + bOffset);

    return col;
}

// --------------------------------------------------------------------------------------------------
// TECHNIQUES:
// --------------------------------------------------------------------------------------------------

technique Levels
{
    pass P0
    {
        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_2_0 psLevels();
    }
}


technique TFixedFunction
{
    pass P0
    {
        //transforms
        WorldTransform[0]   = (tW);
        ViewTransform       = (tV);
        ProjectionTransform = (tP);

        //texturing
        Sampler[0] = (Samp);
        TextureTransform[0] = (tTex);
        TexCoordIndex[0] = 0;
        TextureTransformFlags[0] = COUNT2;
        //Wrap0 = U;  // useful when mesh is round like a sphere

        Lighting       = FALSE;

        //shaders
        VertexShader = NULL;
        PixelShader  = NULL;
    }
}


