//@author: Digital Slaves
//@help: Grid to Sphere
//@tags: morphing

// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------
#define Pi  3.14159
#define MinAngle 0.01

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;         //view matrix as set via Renderer (EX9)
float4x4 tP: PROJECTION;   //projection matrix as set via Renderer (EX9)
float4x4 tWVP: WORLDVIEWPROJECTION;

//material properties
float4 cAmb : COLOR <String uiname="Color";>  = {1, 1, 1, 1};
float Alpha = 1;

//tweakable
float P <
float UIMin = 0.0000;
float UIMax = 1.0000;
> = 0 ;

//texture
texture Tex <string uiname = "Texture" ;>;
sampler Samp = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (Tex);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

float4x4 tTex: TEXTUREMATRIX <string uiname="Texture Transform";>;

//the data structure: vertexshader to pixelshader
//used as output data with the VS function
//and as input data with the PS function
struct vs2ps
{
    float4 Pos : POSITION;
    float4 TexCd : TEXCOORD0;
};

// --------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// --------------------------------------------------------------------------------------------------

vs2ps VS(
    float4 Pos :    POSITION,
	float3 normal:  NORMAL,
    float4 TexCd :  TEXCOORD0)
{
    //inititalize all fields of output struct with 0
    vs2ps Out = (vs2ps)0;
	
	Pos.x = -1 + TexCd.x * 2;
	Pos.y = 1 - TexCd.y * 2;
	Pos.z = 0;
	Pos.w = 1;
	
	float Q;
    Q = 2* P * Pi;
	
	if(Q > Pi) Q = Pi;
	
    float R ;
	R = 1/Q;
	
    float a;
	float b;
	if(Q > MinAngle){
		
		a = -Q + TexCd.x * 2 * Q;
		
		float3 Normal = normalize(normal);
		Normal.x = sin(a);
		Normal.z = -cos(a);
		Pos.x = Normal.x * R ;
		Pos.z = Normal.z * R + cos(Q * 0.5) * R; 
	}

	if(P > 0.5){
		
		a += Pi * 0.5 ;
		b = -Pi + Pi * TexCd.y;

		float3 Para;
		Para.x = cos(a) * sin(b);
		Para.y = -cos(b);
		Para.z = sin(a) * sin(b);

		float f = 2.0 * (P - 0.5);
		Pos.x += f * (Para.x - Pos.x);
		Pos.y += f * (Para.y - Pos.y);
		Pos.z += f * (Para.z - Pos.z);
		
	}

	Out.Pos = mul(Pos, tWVP);
	Out.TexCd = mul(TexCd, tTex);

	return Out;

}

// --------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// --------------------------------------------------------------------------------------------------

float4 PS(vs2ps In): COLOR
{
    //In.TexCd = In.TexCd / In.TexCd.w; // for perpective texture projections (e.g. shadow maps) ps_2_0

    float4 col = tex2D(Samp, In.TexCd) * cAmb;
    col.a *= Alpha;
    return col;
}

// --------------------------------------------------------------------------------------------------
// TECHNIQUES:
// --------------------------------------------------------------------------------------------------

technique TMorph
{
    pass P0
    {
        //Wrap0 = U;  // useful when mesh is round like a sphere
        VertexShader = compile vs_2_0 VS();
        PixelShader = compile ps_2_0 PS();
    }
}

technique TConstantFF
{
    pass P0
    {
        //transformations
        WorldTransform[0]   = (tW);
        ViewTransform       = (tV);
        ProjectionTransform = (tP);

        //material
        MaterialAmbient  = {1, 1, 1, 1};
        MaterialDiffuse  = {1, 1, 1, 1};

        //texturing
        Sampler[0] = (Samp);
        TextureTransform[0] = (tTex);
        TexCoordIndex[0] = 0;
        TextureTransformFlags[0] = COUNT4 | PROJECTED;
        //Wrap0 = U;  // useful when mesh is round like a sphere

        //lighting
        LightEnable[0] = TRUE;
        Lighting       = TRUE;
        SpecularEnable = FALSE;

        LightType[0]     = DIRECTIONAL;
        LightAmbient[0]  = (cAmb);
        LightDiffuse[0]  = {0, 0, 0, 0};
        LightDirection[0] = {0, 0, 1, 1};

        //shading
        ShadeMode = FLAT;
        VertexShader = NULL;
        PixelShader  = NULL;
    }
}