#region usings
using System;
using System.Runtime.InteropServices;
using System.ComponentModel.Composition;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;
using VVVV.Core.Logging;

using NAudio.Wave;
using LibPDBinding;

#endregion usings

namespace VVVV.Nodes
{
	#region PluginInfo
	[PluginInfo(Name = "PDLoader",
	Category = "NAudio",
	Help = "Basic loader of PD patches",
	Tags = "",
	AutoEvaluate = true)]
	#endregion PluginInfo
	public class PDLoaderNode : IPluginEvaluate, IDisposable, IWaveProvider
	{
		#region fields & pins
		[Input("Filename", StringType = StringType.Filename)]
		IDiffSpread<string> FFilename;
		
		[Input("Restart", IsBang=true)]
		ISpread<bool> FRestart;
		
		[Input("Play")]
		IDiffSpread<bool> FPlay;
		
		[Input("Parameter")]
		ISpread<int> FParameter;
		
		[Input("Patch", StringType = StringType.Filename)]
		IDiffSpread<string> FPatchName;
		
		private AsioOut FDriver;
		private WaveStream FStream;
		
		[Import()]
		ILogger FLogger;
		#endregion fields & pins
		
		public void Dispose()
		{
			LibPD.ReInit();
			FDriver.Dispose();
		}
		
		private void RestartAudio()
		{
			if(FDriver != null)
			{
				Dispose();
			}
			
			FDriver = new AsioOut(); 
			FStream = new WaveFileReader(FFilename[0]);
			FStream = new WaveChannel32(FStream);
			 
			LibPD.OpenPatch(FPatchName[0]);
			LibPD.OpenAudio(2, 2, 44100);
			LibPD.ComputeAudio(true);
			
			FDriver.Init(this);
			FDriver.Play();
		}
		
		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			bool needsInit = false;
			if(FPatchName.IsChanged || FFilename.IsChanged)
			{
				RestartAudio();
				needsInit = true;
			}
			
			if (FRestart[0])
				FStream.CurrentTime = TimeSpan.FromSeconds(0);
			
			if (FPlay.IsChanged || needsInit)
				if (FPlay[0])
					FDriver.Play();
				else
					FDriver.Stop();

			LibPD.SendFloat("param", FParameter[0]);
		}
		
		public WaveFormat WaveFormat
		{
			get
			{
				return WaveFormat.CreateIeeeFloatWaveFormat(44100, 2);
			}
		}
		
		public unsafe int Read(byte[] buffer, int offset, int sampleCount)
		{
			FStream.Read(buffer, offset, sampleCount);
			
			var ticks = (sampleCount / (sizeof(float) * WaveFormat.Channels)) / LibPD.BlockSize;
			
			fixed(byte* outBuff = buffer)
			{
				var output = (float*)outBuff;
				LibPD.Process(ticks, output, output);
			}
			
			return sampleCount;
		}
	}
}
