//@author: umanema
//@help: a skybox implementation of a shader from shadertoy
//@tags: skybox, raymarching, shadertoy
//@credits: Inigo Quilez and Pol Jeremies for brilliant Shadertoy.com
// and Bradley Austin's technique of applying Shadertoy to skybox described in
// his book Oculus Rift in Action
//@license: Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

//Look through original code on https://www.shadertoy.com/view/4dfGzs 

//Original shader has 3 iChannels or in other words texture inputs
//Delete unnecessary Textures and SamplerState if you don't need them in your Shaderbox
//Among converted boxes from contribution there are examples with and without them

Texture2D texNoise <string uiname="Noise";>;
Texture2D texAbstract <string uiname="Abstract";>;
Texture2D texLichen <string uiname="Lichen";>;

SamplerState linearSampler : IMMUTABLE
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = Wrap;
    AddressV = Wrap;
};

//Different shaders from shadertoy has different number of variables
//It's a good idea to put all variables in constant buffers
//Look here to know how to manage your constant buffers right https://vvvv.org/documentation/dx11-manage-your-constant-buffers

cbuffer cbControls:register(b0){
	//This is standart hlsl stuff that you don't want to touch
	float4x4 tW:WORLD;
	float4x4 tV:VIEW;
	float4x4 tP:PROJECTION;
	float4x4 tVI:VIEWINVERSE;
	float4x4 tPI:PROJECTIONINVERSE;
	float4x4 tWIT:WORLDINVERSETRANSPOSE;
	
	//Here I've added iTime variable from original shader to animate it
	float iTime <string uiname="iTime";> = 0.0;
};

struct VS_IN{
	float4 PosO:POSITION;
	float4 TexCd:TEXCOORD0;
};

struct VS_OUT{
	float4 PosWVP:SV_POSITION;
	float4 TexCd:TEXCOORD0;
	float4 PosW:TEXCOORD1;
};

VS_OUT VS(VS_IN In){
	VS_OUT Out=(VS_OUT)0;
	float4 PosW=In.PosO;
	PosW=mul(PosW,tW);
	Out.TexCd=In.TexCd;
	Out.PosW=PosW;
	Out.PosWVP=mul(PosW,mul(tV,tP));
	return Out;
}

//I've copied the whole code from Shadertoy below
//You can read how to port GLSL to HLSL on
//https://docs.microsoft.com/en-us/windows/uwp/gaming/glsl-to-hlsl-reference
//I've commented what I've changed
//Though main principles remain it will be different for your shader

//GLSL vector types vec2, vec3, vec4 is equivalent to float2, float3, float4 in HLSL
float noise( in float3 x )
{
    float3 p = floor(x);
    //Changed fract to frac
	float3 f = frac(x);
	f = f*f*(3.0-2.0*f);
	float2 uv = (p.xy+float2(37.0,17.0)*p.z) + f.xy;
	//GLSL function gvec4 textureLod(gsampler2D sampler, vec2 P, float lod)
	//is equivalent to Object.SampleLevel( sampler_state S, float Location, float LOD [, int Offset] )
	float2 rg = texNoise.SampleLevel( linearSampler, (uv+0.5)/256.0, 0.0 ).xy;
	//Changed mix to lerp
	return lerp( rg.x, rg.y, f.z );
}

//Sampler2D is Texture2D in HLSL
float4 texcube(  Texture2D tex , in float3 p, in float3 n )
{
    float3 m = abs( n );
	float4 x = tex.SampleLevel( linearSampler, p.yz, 0.0);
	float4 y = tex.SampleLevel( linearSampler, p.zx, 0.0);
	float4 z = tex.SampleLevel( linearSampler, p.xy, 0.0);
	return x*m.x + y*m.y + z*m.z;
}

float mapTerrain( float3 p )
{
	p *= 0.1; 
	p.xz *= 0.6;
	
	float time = 0.5 + 0.15*iTime;
	float ft = frac( time );
	float it = floor( time );
	ft = smoothstep( 0.7, 1.0, ft );
	time = it + ft;
	float spe = 1.4;
	
	float f;
    f  = 0.5000*noise( p*1.00 + float3(0.0,1.0,0.0)*spe*time );
    f += 0.2500*noise( p*2.02 + float3(0.0,2.0,0.0)*spe*time );
    f += 0.1250*noise( p*4.01 );
	return 25.0*f-10.0;
}

static float3 gro = float3(0.0,0.0,0.0);

float map(in float3 c) 
{
	float3 p = c + 0.5;
	
	float f = mapTerrain( p ) + 0.25*p.y;

    f = lerp( f, 1.0, step( length(gro-p), 5.0 ) );

	return step( f, 0.5 );
}

float3 lig = normalize( float3(-0.4,0.3,0.7) );

float castRay( in float3 ro, in float3 rd, out float3 oVos, out float3 oDir )
{
	float3 pos = floor(ro);
	float3 ri = 1.0/rd;
	float3 rs = sign(rd);
	float3 dis = (pos-ro + 0.5 + rs*0.5) * ri;
	
	float res = -1.0;
	float3 mm = float3(0.0,0.0,0.0);
	for( int i=0; i<128; i++ ) 
	{
		if( map(pos)>0.5 ) { res=1.0; break; }
		mm = step(dis.xyz, dis.yxy) * step(dis.xyz, dis.zzx);
		dis += mm * rs * ri;
        pos += mm * rs;
	}

	float3 nor = -mm*rs;
	float3 vos = pos;
		
	float3 mini = (pos-ro + 0.5 - 0.5*float3(rs))*ri;
	float t = max ( mini.x, max ( mini.y, mini.z ) );
	
	oDir = mm;
	oVos = vos;

	return t*res;
}

float3 path( float t, float ya )
{
    float2 p  = 100.0*sin( 0.02*t*float2(1.0,1.2) + float2(0.1,0.9) );
	       p +=  50.0*sin( 0.04*t*float2(1.3,1.0) + float2(1.0,4.5) );
	
	return float3( p.x, 18.0 + ya*4.0*sin(0.05*t), p.y );
}

//mat2, mat3, mat4 should be float2x2, float3x3, float4x4
float3x3 setCamera( in float3 ro, in float3 ta, float cr )
{
	float3 cw = normalize(ta-ro);
	float3 cp = float3(sin(cr), cos(cr),0.0);
	float3 cu = normalize( cross(cw,cp) );
	float3 cv = normalize( cross(cu,cw) );
    return float3x3( cu, cv, -cw );
}

float maxcomp( in float4 v )
{
    return max( max(v.x,v.y), max(v.z,v.w) );
}

float isEdge( in float2 uv, float4 va, float4 vb, float4 vc, float4 vd )
{
    float2 st = 1.0 - uv;

    float4 wb = smoothstep( 0.85, 0.99, float4(uv.x,
											   st.x,
											   uv.y,
											   st.y) ) * ( 1.0 - va + va*vc );
	
    float4 wc = smoothstep( 0.85, 0.99, float4(uv.x*uv.y,
											   st.x*uv.y,
											   st.x*st.y,
											   uv.x*st.y) ) * ( 1.0 - vb + vd*vb );
    return maxcomp( max(wb,wc) );
}

float calcOcc( in float2 uv, float4 va, float4 vb, float4 vc, float4 vd )
{
    float2 st = 1.0 - uv;

    float4 wa = float4( uv.x, st.x, uv.y, st.y ) * vc;

    float4 wb = float4(uv.x*uv.y,
					   st.x*uv.y,
					   st.x*st.y,
					   uv.x*st.y)*vd*(1.0-vc.xzyw)*(1.0-vc.zywx);
    
    return wa.x + wa.y + wa.z + wa.w +
           wb.x + wb.y + wb.z + wb.w;
}

float3 render( in float3 ro, in float3 rd )
{
    float3 col = float3(0.0,0.0,0.0);
	
	float3 vos, dir;
	float t = castRay( ro, rd, vos, dir );
	if( t>0.0 )
	{
        float3 nor = -dir*sign(rd);
        float3 pos = ro + rd*t;
        float3 uvw = pos - vos;
		
		float3 v1  = vos + nor + dir.yzx;
	    float3 v2  = vos + nor - dir.yzx;
	    float3 v3  = vos + nor + dir.zxy;
	    float3 v4  = vos + nor - dir.zxy;
		float3 v5  = vos + nor + dir.yzx + dir.zxy;
        float3 v6  = vos + nor - dir.yzx + dir.zxy;
	    float3 v7  = vos + nor - dir.yzx - dir.zxy;
	    float3 v8  = vos + nor + dir.yzx - dir.zxy;
	    float3 v9  = vos + dir.yzx;
	    float3 v10 = vos - dir.yzx;
	    float3 v11 = vos + dir.zxy;
	    float3 v12 = vos - dir.zxy;
 	    float3 v13 = vos + dir.yzx + dir.zxy; 
	    float3 v14 = vos - dir.yzx + dir.zxy ;
	    float3 v15 = vos - dir.yzx - dir.zxy;
	    float3 v16 = vos + dir.yzx - dir.zxy;

		float4 vc = float4( map(v1),  map(v2),  map(v3),  map(v4)  );
	    float4 vd = float4( map(v5),  map(v6),  map(v7),  map(v8)  );
	    float4 va = float4( map(v9),  map(v10), map(v11), map(v12) );
	    float4 vb = float4( map(v13), map(v14), map(v15), map(v16) );
		
		float2 uv = float2( dot(dir.yzx, uvw), dot(dir.zxy, uvw) );

        float www = 1.0 - isEdge( uv, va, vb, vc, vd );
        
        float3 wir = smoothstep( 0.4, 0.5, abs(uvw-0.5) );
        float vvv = (1.0-wir.x*wir.y)*(1.0-wir.x*wir.z)*(1.0-wir.y*wir.z);

        col = 2.0 * texAbstract.SampleLevel( linearSampler, 0.01*pos.xz, 0.0 ).zyx; 
        col += 0.8 * float3(0.1,0.3,0.4);
        col *= 0.5 + 0.5*texcube( texLichen, 0.5*pos, nor ).x;
        col *= 1.0 - 0.75*(1.0-vvv)*www;
		
        float dif = clamp( dot( nor, lig ), 0.0, 1.0 );
        float bac = clamp( dot( nor, normalize(lig*float3(-1.0,0.0,-1.0)) ), 0.0, 1.0 );
        float sky = 0.5 + 0.5*nor.y;
        float amb = clamp(0.75 + pos.y/25.0,0.0,1.0);
        float occ = 1.0;

        occ = calcOcc( uv, va, vb, vc, vd );
        occ = 1.0 - occ/8.0;
        occ = occ*occ;
        occ = occ*occ;
        occ *= amb;

        float3 lin = float3(0.0,0.0,0.0);
        lin += 2.5*dif*float3(1.00,0.90,0.70)*(0.5+0.5*occ);
        lin += 0.5*bac*float3(0.15,0.10,0.10)*occ;
        lin += 2.0*sky*float3(0.40,0.30,0.15)*occ;
	
        float lineglow = 0.0;
        lineglow += smoothstep( 0.4, 1.0,     uv.x )*(1.0-va.x*(1.0-vc.x));
        lineglow += smoothstep( 0.4, 1.0, 1.0-uv.x )*(1.0-va.y*(1.0-vc.y));
        lineglow += smoothstep( 0.4, 1.0,     uv.y )*(1.0-va.z*(1.0-vc.z));
        lineglow += smoothstep( 0.4, 1.0, 1.0-uv.y )*(1.0-va.w*(1.0-vc.w));
        lineglow += smoothstep( 0.4, 1.0,      uv.y*      uv.x )*(1.0-vb.x*(1.0-vd.x));
        lineglow += smoothstep( 0.4, 1.0,      uv.y* (1.0-uv.x))*(1.0-vb.y*(1.0-vd.y));
        lineglow += smoothstep( 0.4, 1.0, (1.0-uv.y)*(1.0-uv.x))*(1.0-vb.z*(1.0-vd.z));
        lineglow += smoothstep( 0.4, 1.0, (1.0-uv.y)*     uv.x )*(1.0-vb.w*(1.0-vd.w));
		
        float3 linCol = 2.0*float3(5.0,0.6,0.0);
        linCol *= (0.5+0.5*occ)*0.5;
        lin += 3.0*lineglow*linCol;
		
        col = col*lin;
        col += 8.0*linCol*float3(1.0,2.0,3.0)*(1.0-www);//*(0.5+1.0*sha);
        col += 0.1*lineglow*linCol;
        col *= min(0.1,exp( -0.07*t ));
			
        float3 col2 = float3(1.3,1.3,1.3)*(0.5+0.5*nor.y)*occ*www*(0.9+0.1*vvv)*exp( -0.04*t );;
        float mi = sin(-1.57+0.5*iTime);
        mi = smoothstep( 0.70, 0.75, mi );
        col = lerp( col, col2, mi );
	}

	// gamma	
	col = pow( col, float3(0.45,0.45,0.45) );

    return col;
}

//This is our main function
//And the code from mainImage function goes here
float4 psDefaultbox(VS_OUT In):SV_Target{	
	
	//The first string is used to 
	//vec2 q = fragCoord.xy / iResolution.xy;
	float2 q = 1-In.TexCd.rg;
	float2 p = -1.0 + 2.0*q;
    
	//p.x *= iResolution.x/ iResolution.y;
	
	//We don't need mouse input anymore
    //vec2 mo = iMouse.xy / iResolution.xy;
    //if( iMouse.w<=0.00001 ) mo=vec2(0.0);
	//float time = 2.0*iTime + 50.0*mo.x;
	//p.x *= TargetSize.x / TargetSize.y;
	
	float time = 2.0*iTime;
	
	float cr = 0.2*cos(0.1*iTime);
	float3 ro = path( time+0.0, 1.0 );
	float3 ta = path( time+5.0, 1.0 ) - float3(0.0,6.0,0.0);
	gro = ro;

    float3x3 cam = setCamera( ro, ta, cr );
	
    float r2 = p.x*p.x*0.32 + p.y*p.y;
    p *= (7.0-sqrt(37.5-11.5*r2))/(r2+1.0);
	
	////////////////////////////////////////////////////////////////////////////
	//The main thing to adapt Shadertoy to VR is to interpolate the ray
	//directions across every pixel of input geometry.
	//Each time you have find 3-dimensional vector called rd or anyhow it is called and
	//assign normalize( In.PosW.xyz ) to it
	////////////////////////////////////////////////////////////////////////////
    float3 rd = normalize( In.PosW.xyz ); //<--All the magic is here

    float3 col = render( ro, rd );
    
	//Often you need to get rid of postprocessing effect
	//especially vingetting otherwise you'll see the box edges
	//col *= 0.5 + 0.5*pow( 16.0*q.x*q.y*(1.0-q.x)*(1.0-q.y), 0.1 );
	
	//fragColor is what you should return
	//fragColor = vec4( col, 1.0 );
	return float4( col, 1.0 );
}

technique10 Default{
	pass P0{
		SetVertexShader(CompileShader(vs_4_0,VS()));
		SetPixelShader(CompileShader(ps_4_0,psDefaultbox()));
	}
}


