//@author: umanema
//@help: a skybox implementation of a shader from shadertoy
//@tags: skybox, raymarching, shadertoy
//@credits: inigo quilez - iq/2013, sebl
//@license: Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

float2 TargetSize = {512.0, 512.0};

Texture2D tex0 <string uiname="NoiseTexture";>;
Texture2D tex1 <string uiname="Texture";>;

SamplerState s0 
{
	Filter = MIN_POINT_MAG_LINEAR_MIP_POINT;
	AddressU = wrap;
	AddressV = wrap;
};

cbuffer cbControls:register(b0){
	float4x4 tW:WORLD;
	float4x4 tV:VIEW;
	float4x4 tP:PROJECTION;
	float4x4 tVI:VIEWINVERSE;
	float4x4 tPI:PROJECTIONINVERSE;
	float4x4 tWIT:WORLDINVERSETRANSPOSE;
	
	float Time;
	
};

struct VS_IN{
	float4 PosO:POSITION;
	float4 TexCd:TEXCOORD0;
};

struct VS_OUT{
	float4 PosWVP:SV_POSITION;
	float4 TexCd:TEXCOORD0;
	float4 PosW:TEXCOORD1;
};

VS_OUT VS(VS_IN In){
	VS_OUT Out=(VS_OUT)0;
	float4 PosW=In.PosO;
	PosW=mul(PosW,tW);
	Out.PosW=PosW;
	Out.TexCd = In.TexCd;
	Out.PosWVP=mul(PosW,mul(tV,tP));
	return Out;
}

#define SC (250.0)

/*******************************************************************************
NOISE
*******************************************************************************/

float3 noised( in float2 x )
{
    float2 p = floor(x);
    float2 f = frac(x);
    float2 u = f*f*(3.0-2.0*f);
	float a = tex0.SampleLevel(s0,(p+float2(0.5,0.5))/256.0,-100.0).x;
	float b = tex0.SampleLevel(s0,(p+float2(1.5,0.5))/256.0,-100.0).x;
	float c = tex0.SampleLevel(s0,(p+float2(0.5,1.5))/256.0,-100.0).x;
	float d = tex0.SampleLevel(s0,(p+float2(1.5,1.5))/256.0,-100.0).x;
	return float3(a+(b-a)*u.x+(c-a)*u.y+(a-b-c+d)*u.x*u.y,
				6.0*f*(1.0-f)*(float2(b-a,c-a)+(a-b-c+d)*u.yx));
}

/*******************************************************************************
METHODS
*******************************************************************************/

float2x2 m2 = float2x2(0.8,-0.6,
		     		   0.6,0.8);

float detailH( in float2 x )
{
    float d = 0.0;//50.0*tex1.SampleSampleLevel( s0, x*0.03/SC, 0.0 ).x;
    return d + 0.5*tex1.SampleLevel( s0, x*2.0/SC, 0.0 ).x;
}

float detailM( in float2 x )
{
    float d = 0.0;//50.0*tex1.SampleSampleLevel( s0, x*0.03/SC, 0.0 ).x;
    return d;
}

float terrainH( in float2 x )
{
	float2  p = x*0.003/SC;
    float a = 0.0;
    float b = 1.0;
	float2  d = float(0.0);
    for( int i=0; i<15; i++ )
    {
        float3 n = noised(p);
        d += n.yz;
        a += b*n.x/(1.0+dot(d,d));
		b *= 0.5;
        p = mul(m2,p*2.0);  /////!!!!///////
    }

    float de = detailH(x);
	return SC*100.0*a - de;
}

float terrainM( in float2 x )
{
	float2  p = x*0.003/SC;
    float a = 0.0;
    float b = 1.0;
	float2  d = float(0.0);
    for( int i=0; i<9; i++ )
    {
        float3 n = noised(p);
        d += n.yz;
        a += b*n.x/(1.0+dot(d,d));
		b *= 0.5;
        p = mul(m2,p*2.0);    /////!!!!///////
    }
	return SC*100.0*a - detailH(x);
}

float terrainL( in float2 x )
{
	float2  p = x*0.003/SC;
    float a = 0.0;
    float b = 1.0;
	float2  d = float(0.0);
    for( int i=0; i<7; i++ )
    {
        float3 n = noised(p);
        d += n.yz;
        a += b*n.x/(1.0+dot(d,d));
		b *= 0.5;
        p = mul(m2,p*2.0);
    }

	return SC*100.0*a;
}

float intersect( in float3 ro, in float3 rd, in float tmin, in float tmax )
{
    float t = tmin;
	for( int i=0; i<256; i++ )
	{
        float3 pos = ro + t*rd;
		float h = pos.y - terrainM( pos.xz );
		if( h<(0.002*t) || t>tmax ) break;
		t += 0.5*h;
	}

	return t;
}

float softShadow(in float3 ro, in float3 rd )
{
    // real shadows	
    float res = 1.0;
    float t = 0.001;
	for( int i=0; i<80; i++ )
	{
	    float3  p = ro + t*rd;
        float h = p.y - terrainM( p.xz );
		res = min( res, 16.0*h/t );
		t += h;
		if( res<0.001 ||p.y>(SC*200.0) ) break;
	}
	return clamp( res, 0.0, 1.0 );
}

float3 calcNormal( in float3 pos, float t )
{
    float2  eps = float2( 0.002*t, 0.0 );
    return normalize( float3( terrainH(pos.xz-eps.xy) - terrainH(pos.xz+eps.xy),
                            2.0*eps.x,
                            terrainH(pos.xz-eps.yx) - terrainH(pos.xz+eps.yx) ) );
}

float3 camPath( float time )
{
	return SC*1100.0*float3( cos(0.0+0.23*time), 0.0, cos(1.5+0.21*time) );
}
	
float fbm( float2 p )
{
    float f = 0.0;
    f += 0.5000*tex0.Sample (s0, p/256.0 ).x; p = mul(m2,p*2.02); /////!!!////:
    f += 0.2500*tex0.Sample (s0, p/256.0 ).x; p = mul(m2,p*2.03);
    f += 0.1250*tex0.Sample ( s0, p/256.0 ).x; p = mul(m2,p*2.01);
    f += 0.0625*tex0.Sample ( s0, p/256.0 ).x;
    return f/0.9375;
}

float3x3 setCamera( in float3 ro, in float3 ta, in float cr )
{
	float3 cw = normalize(ta-ro);
	float3 cp = float3(sin(cr), cos(cr),0.0);
	float3 cu = normalize( cross(cw,cp) );
	float3 cv = normalize( cross(cu,cw) );
    return float3x3( cu, cv, cw );
}

float3 render( in float3 ro, in float3 rd )
{
    float3 light1 = normalize( float3(-0.8,0.4,-0.3) );
    // bounding plane
    float tmin = 1.0;
    float tmax = 1000.0*SC;
#if 1
    float maxh = 300.0*SC;
    float tp = (maxh-ro.y)/rd.y;
    if( tp>0.0 )
    {
        if( ro.y>maxh ) tmin = max( tmin, tp );
        else            tmax = min( tmax, tp );
    }
#endif
	float sundot = clamp(dot(rd,light1),0.0,1.0);
	float3 col;
    float t = intersect( ro, rd, tmin, tmax );
    if( t>tmax)
    {
        // sky		
		col = float3(0.3,.55,0.8)*(1.0-0.8*rd.y)*0.9;
        // sun
		col += 0.25*float3(1.0,0.7,0.4)*pow( sundot,5.0 );
		col += 0.25*float3(1.0,0.8,0.6)*pow( sundot,64.0 );
		col += 0.2*float3(1.0,0.8,0.6)*pow( sundot,512.0 );
        // clouds
		float2 sc = ro.xz + rd.xz*(SC*1000.0-ro.y)/rd.y;
		col = lerp( col, float3(1.0,0.95,1.0), 0.5*smoothstep(0.5,0.8,fbm(0.0005*sc/SC)) );
        // horizon
        col = lerp( col, float3(0.7,0.75,0.8), pow( 1.0-max(rd.y,0.0), 8.0 ) );
	}
	else
	{
        // mountains		
		float3 pos = ro + t*rd;
        float3 nor = calcNormal( pos, t );
        //nor = normalize( nor + 0.5*( float3(-1.0,0.0,-1.0) + float3(2.0,1.0,2.0)*texture2D(iChannel1,0.01*pos.xz).xyz) );
        float3 ref = reflect( rd, nor );
        float fre = clamp( 1.0+dot(rd,nor), 0.0, 1.0 );
        
        // rock
		float r = tex0.Sample( s0, (7.0/SC)*pos.xz/256.0 ).x;
        col = (r*0.25+0.75)*0.9*lerp( float3(0.08,0.05,0.03), float3(0.10,0.09,0.08), 
                                     tex0.Sample(s0,0.00007*float2(pos.x,pos.y*48.0)/SC).x );
		col = lerp( col, 0.20*float3(0.45,.30,0.15)*(0.50+0.50*r),smoothstep(0.70,0.9,nor.y) );
        col = lerp( col, 0.15*float3(0.30,.30,0.10)*(0.25+0.75*r),smoothstep(0.95,1.0,nor.y) );

		// snow
		float h = smoothstep(55.0,80.0,pos.y/SC + 25.0*fbm(0.01*pos.xz/SC) );
        float e = smoothstep(1.0-0.5*h,1.0-0.1*h,nor.y);
        float o = 0.3 + 0.7*smoothstep(0.0,0.1,nor.x+h*h);
        float s = h*e*o;
        col = lerp( col, 0.29*float3(0.62,0.65,0.7), smoothstep( 0.1, 0.9, s ) );
		
         // lighting		
        float amb = clamp(0.5+0.5*nor.y,0.0,1.0);
		float dif = clamp( dot( light1, nor ), 0.0, 1.0 );
		float bac = clamp( 0.2 + 0.8*dot( normalize( float3(-light1.x, 0.0, light1.z ) ), nor ), 0.0, 1.0 );
		float sh = 1.0; if( dif>=0.0001 ) sh = softShadow(pos+light1*20.0,light1);
		
		float3 lin  = float(0.0);
		lin += dif*float3(7.00,5.00,3.00)*float3( sh, sh*sh*0.5+0.5*sh, sh*sh*0.8+0.2*sh );
		lin += amb*float3(0.40,0.60,0.80)*1.2;
        lin += bac*float3(0.40,0.50,0.60);
		col *= lin;
        
        col += s*0.1*pow(fre,4.0)*float3(7.0,5.0,3.0)*sh * pow( clamp(dot(light1,ref), 0.0, 1.0),16.0);
        col += s*0.1*pow(fre,4.0)*float3(0.4,0.5,0.6)*smoothstep(0.0,0.6,ref.y);

		// fog
        //float fo = 1.0-exp(-0.000004*t*t/(SC*SC) );
        float fo = 1.0-exp(-0.001*t/SC );
        float3 fco = 0.7*float3(0.5,0.7,0.9) + 0.1*float3(1.0,0.8,0.5)*pow( sundot, 4.0 );
		col = lerp( col, fco, fo );

        // sun scatter
		col += 0.3*float3(1.0,0.8,0.4)*pow( sundot, 8.0 )*(1.0-exp(-0.002*t/SC));
	}

    // gamma
	col = pow(abs(col),float(0.4545));
	
	return col;
}

float4 psElevated(VS_OUT In): SV_Target
{
	float2 xy = 1.0 - 2.0*In.TexCd.xy;
	//float xy.y=1-
	float2 s = xy*float2(TargetSize.x/TargetSize.y,1.0);

    float time = Time*0.15 + 0.3 + 4.0;

    // camera position
	float3 ro = camPath( time );
	float3 ta = camPath( time + 3.0 );
	ro.y = terrainL( ro.xz ) + 11.0*SC;
	ta.y = ro.y - 20.0*SC;
	float cr = 0.2*cos(0.1*time);
    
//ro = float3(0.0,0.0,0.0); ro += camPath(20.0 ); ro.y += terrainM( ro.xz ) + 4.0; cr = 0.0; ta = ro + float3(0.0,0.0,-1.0);    
    
    // camera2world transform    
    float3x3 cam = setCamera( ro, ta, cr );

    // camera ray    
	float3 rd = normalize(In.PosW.xyz);
	
    float3 col = render( ro, rd );

    return float4( col, 1.0 );
}


technique10 Elevated{
	pass P0{
		SetVertexShader(CompileShader(vs_4_0,VS()));
		SetPixelShader(CompileShader(ps_4_0,psElevated()));
	}
}

