//@author: umanema
//@help: a skybox implementation of a shader from shadertoy
//@tags: skybox, raymarching, shadertoy
//@credits: Shane
//@license: Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

Texture2D tex <string uiname="Texture";>;

SamplerState s0 
{
	Filter = MIN_POINT_MAG_LINEAR_MIP_POINT;
	AddressU = wrap;
	AddressV = wrap;
};

static const float FAR = 50.0; // Far plane.

cbuffer cbControls:register(b0){
	float4x4 tW:WORLD;
	float4x4 tV:VIEW;
	float4x4 tP:PROJECTION;
	float4x4 tVI:VIEWINVERSE;
	float4x4 tPI:PROJECTIONINVERSE;
	float4x4 tWIT:WORLDINVERSETRANSPOSE;
	float iTime = 0.0;
	static float objID = 0.;
	
};

struct VS_IN{
	float4 PosO:POSITION;
	float4 TexCd:TEXCOORD0;
};

struct VS_OUT{
	float4 PosWVP:SV_POSITION;
	float4 TexCd:TEXCOORD0;
	float4 PosW:TEXCOORD1;
};

VS_OUT VS(VS_IN In){
	VS_OUT Out=(VS_OUT)0;
	float4 PosW=In.PosO;
	PosW=mul(PosW,tW);
	Out.TexCd = In.TexCd;
	Out.PosW=PosW;
	Out.PosWVP=mul(PosW,mul(tV,tP));
	return Out;
}

// Simple hash function.
float hash( float n ){ return frac(cos(n)*45758.5453); }



// Tri-Planar blending function. Based on an old Nvidia writeup:
// GPU Gems 3 - Ryan Geiss: https://developer.nvidia.com/gpugems/GPUGems3/gpugems3_ch01.html
float3 tex3D(SamplerState t, in float3 p, in float3 n ){
    
    n = max(abs(n), 0.001);
    n /= dot(n, float3(1.0,1.0,1.0));
	float3 tx = tex.SampleLevel(t, p.yz, 0.0).xyz;
    float3 ty = tex.SampleLevel(t, p.zx, 0.0).xyz;
    float3 tz = tex.SampleLevel(t, p.xy, 0.0).xyz;
    
    // Textures are stored in sRGB (I think), so you have to convert them to linear space 
    // (squaring is a rough approximation) prior to working with them... or something like that. :)
    // Once the final color value is gamma corrected, you should see correct looking colors.
    return (tx*tx*n.x + ty*ty*n.y + tz*tz*n.z);
    
}

// Common formula for rounded squares, for all intended purposes.
float lengthN(in float2 p, in float n){
	p = pow(abs(p), float2(n,n));
	return pow(p.x + p.y, 1.0/n);
}


// The camera path: There are a few spline setups on Shadertoy, but this one is a slight variation of
// Otavio Good's spline setup in his "Alien Beacon" shader: https://www.shadertoy.com/view/ld2SzK
//
// Spline point markers ("cp" for camera point). The camera visits each point in succession, then loops
// back to the first point, when complete, in order to repeat the process. In case it isn't obvious, each 
// point represents an open space juncture in the object that links to the previous and next point.
// Of course, running a camera in a straight line between points wouldn't produce a smooth camera effect, 
// so we apply the Catmull-Rom equation to the line segment.
static float3 cp[16];

void setCamPath(){
    
    // The larger fractal object has nodes in a 4x4x4 grid.
    // The smaller one in a 2x2x2 grid. The following points
    // map a path to various open areas throughout the object.
    const float sl = 2.*.96;
    const float bl = 4.*.96;

    cp[0]  = float3(0, 0, 0);
    cp[1]  = float3(0, 0, bl);
    cp[2]  = float3(sl, 0, bl);
    cp[3]  = float3(sl, 0, sl);
    cp[4]  = float3(sl, sl, sl);	
    cp[5]  = float3(-sl, sl, sl);	
    cp[6]  = float3(-sl, 0, sl);
    cp[7]  = float3(-sl, 0, 0);
    
    cp[8]  = float3(0, 0, 0);	
    cp[9]  = float3(0, 0, -bl);
    cp[10] = float3(0, bl, -bl);	
    cp[11] = float3(-sl, bl, -bl);
    cp[12] = float3(-sl, 0, -bl);
    cp[13] = float3(-sl, 0, 0);
    cp[14] = float3(-sl, -sl, 0);
    cp[15] = float3(0, -sl, 0); 
    
    // Tighening the radius a little, so that the camera doesn't hit the walls.
    // I should probably hardcode this into the above... Done.
    //for(int i=0; i<16; i++) cp[i] *= .96;
    
}

// Standard Catmull-Rom equation. The equation takes in the line segment end points (p1 and p2), the
// points on either side (p0 and p3), the current fractional distance (t) along the segment, then
// returns the the smooth (cubic interpolated) position. The end result is a smooth transition 
// between points... Look up a diagram on the internet. That should make it clearer.
float3 Catmull(float3 p0, float3 p1, float3 p2, float3 p3, float t){

    return (((-p0 + p1*3. - p2*3. + p3)*t*t*t + (p0*2. - p1*5. + p2*4. - p3)*t*t + (-p0 + p2)*t + p1*2.)*.5);
   
}
 
// Camera path. Determine the segment number (segNum), and how far - timewise - we are along it (segTime).
// Feed the segment, the appropriate adjoining segments, and the segment time into the Catmull-Rom
// equation to produce a camera position. The process is pretty simple, once you get the hang of it.
float3 camPath(float t){
    
    const int aNum = 16;
    
    t = frac(t/float(aNum))*float(aNum);	// Repeat every 16 time units.
    
    // Segment number. Range: [0, 15], in this case.
    float segNum = floor(t);
    // Segment portion. Analogous to how far we are alone the individual line segment. Range: [0, 1].
    float segTime = t - segNum; 
    
    
    if (segNum == 0.) return Catmull(cp[aNum-1], cp[0], cp[1], cp[2], segTime); 
    
    for(int i=1; i<aNum-2; i++){
        if (segNum == float(i)) return Catmull(cp[i-1], cp[i], cp[i+1], cp[i+2], segTime); 
    }
    
    if (segNum == float(aNum-2)) return Catmull(cp[aNum-3], cp[aNum-2], cp[aNum-1], cp[0], segTime); 
    if (segNum == float(aNum-1)) return Catmull(cp[aNum-2], cp[aNum-1], cp[0], cp[1], segTime);

    return float3(0.0,0.0,0.0);
    
}

// Smooth minimum function. There are countless articles, but IQ explains it best here:
// http://iquilezles.org/www/articles/smin/smin.htm
float sminP( float a, float b, float s ){

    float h = clamp( 0.5+0.5*(b-a)/s, 0.0, 1.0 );
    return lerp( b, a, h ) - s*h*(1.0-h);
}

// Creating the scene geometry.
//
// There are two intertwined fractal objects. One is a gold and timber lattice, spread out in a 4x4x4
// grid. The second is some metallic tubing spread out over a 2x2x2 grid. Each are created by combining
// repeat objects with various operations. All of it is pretty standard.
//
// The code is a little fused together, in order to save some cycles, but if you're interested in the 
// process, I have a "Menger Tunnel" example that's a little easier to decipher.
float map(in float3 q){

    
///////////

    // The grey section. I have another Menger example, if you'd like to look into that more closely.
    // Layer one.
 	float3 p = abs(frac(q/4.)*4. - 2.);
 	float tube = min(max(p.x, p.y), min(max(p.y, p.z), max(p.x, p.z))) - 4./3. - .015;// + .05;
    

    // Layer two.
    p = abs(frac(q/2.)*2. - 1.);
 	//d = max(d, min(max(p.x, p.y), min(max(p.y, p.z), max(p.x, p.z))) - s/3.);// + .025
 	tube = max(tube, sminP(max(p.x, p.y), sminP(max(p.y, p.z), max(p.x, p.z), .05), .05) - 2./3.);// + .025
   
///////
    // The gold and timber paneling.
    //
    // A bit of paneling, using a combination of repeat objects. We're doing it here in layer two, just
    // to save an extra "fract" call. Very messy, but saves a few cycles... maybe.
    
    //float panel = sminP(length(p.xy),sminP(length(p.yz),length(p.xz), 0.25), 0.125)-0.45; // EQN 1
    //float panel = sqrt(min(dot(p.xy, p.xy),min(dot(p.yz, p.yz),dot(p.xz, p.xz))))-0.5; // EQN 2
    //float panel = min(max(p.x, p.y),min(max(p.y, p.z),max(p.x, p.z)))-0.5; // EQN 3
    float panel = sminP(max(p.x, p.y),sminP(max(p.y, p.z),max(p.x, p.z), .125), .125)-0.5; // EQN 3
    

    // Gold strip. Probably not the best way to do this, but it gets the job done.
    // Identifying the gold strip region, then edging it out a little... for whatever reason. :)
    float strip = step(p.x, .75)*step(p.y, .75)*step(p.z, .75);
    panel -= (strip)*.025;     
    
    // Timber bulge. Just another weird variation.
    //float bulge = (max(max(p.x, p.y), p.z) - .55);//length(p)-1.;//
    //panel -= bulge*(1.-step(p.x, .75)*step(p.y, .75)*step(p.z, .75))*bulge*.25;    
    
    // Repeat field entity two, which is just an abstract object repeated every half unit. 
    p = abs(frac(q*2.)*.5 - .25);
    float pan2 = min(p.x, min(p.y,p.z))-.05;    
    
    // Combining the two entities above.
    panel = max(abs(panel), abs(pan2)) - .0425;    
/////////
    
    // Layer three. 3D space is divided by three.
    p = abs(frac(q*1.5)/1.5 - 1./3.);
 	tube = max(tube, min(max(p.x, p.y), min(max(p.y, p.z), max(p.x, p.z))) - 2./9. + .025); // + .025 


    // Layer three. 3D space is divided by two, instead of three, to give some variance.
    p = abs(frac(q*3.)/3. - 1./6.);
 	tube = max(tube, min(max(p.x, p.y), min(max(p.y, p.z), max(p.x, p.z))) - 1./9. - .035); //- .025 
    

    
    
    // Object ID: Equivalent to: if(tube<panel)objID=2; else objID = 1.; //etc.
    //
    // By the way, if you need to identify multiple objects, you're better off doing it in a seperate pass, 
    // after the raymarching function. Having multiple "if" statements in a distance field equation can 
    // slow things down considerably.
        
    //objID = 2. - step(tube, panel) + step(panel, tube)*(strip);
    objID = 1.+ step(tube, panel) + step(panel, tube)*(strip)*2.;
    //objID = 1. + step(panel, tube)*(strip) + step(tube, panel)*2.;
    

    return min(panel, tube);
    
    
}

float trace(in float3 ro, in float3 rd){

    float t = 0.0, h;
    for(int i = 0; i < 92; i++){
    
        h = map(ro+rd*t);
        // Note the "t*b + a" addition. Basically, we're putting less emphasis on accuracy, as
        // "t" increases. It's a cheap trick that works in most situations... Not all, though.
        if(abs(h)<0.001*(t*.25 + 1.) || t>FAR) break; // Alternative: 0.001*max(t*.25, 1.)
        t += h*.8;
        
    }

    return t;
}


// The reflections are pretty subtle, so not much effort is being put into them. Only eight iterations.
float refTrace(float3 ro, float3 rd){

    float t = 0.0;
    for(int i=0; i<16; i++){
        float d = map(ro + rd*t);
        if (d < 0.0025*(t*.25 + 1.) || t>FAR) break;
        t += d;
    } 
    return t;
}



/*
// Tetrahedral normal, to save a couple of "map" calls. Courtesy of IQ.
vec3 calcNormal(in vec3 p){

    // Note the slightly increased sampling distance, to alleviate artifacts due to hit point inaccuracies.
    vec2 e = vec2(0.0025, -0.0025); 
    return normalize(e.xyy * map(p + e.xyy) + e.yyx * map(p + e.yyx) + e.yxy * map(p + e.yxy) + e.xxx * map(p + e.xxx));
}
*/

// Standard normal function. It's not as fast as the tetrahedral calculation, but more symmetrical. Due to 
// the intricacies of this particular scene, it's kind of needed to reduce jagged effects.
float3 calcNormal(in float3 p) {
	const float2 e = float2(0.005, 0);
	return normalize(float3(map(p + e.xyy) - map(p - e.xyy), map(p + e.yxy) - map(p - e.yxy),	map(p + e.yyx) - map(p - e.yyx)));
}

// I keep a collection of occlusion routines... OK, that sounded really nerdy. :)
// Anyway, I like this one. I'm assuming it's based on IQ's original.
float calcAO(in float3 pos, in float3 nor)
{
	float sca = 2.0, occ = 0.0;
    for( int i=0; i<5; i++ ){
    
        float hr = 0.01 + float(i)*0.5/4.0;        
        float dd = map(nor * hr + pos);
        occ += (hr - dd)*sca;
        sca *= 0.7;
    }
    return clamp( 1.0 - occ, 0.0, 1.0 );    
}


// Texture bump mapping. Four tri-planar lookups, or 12 texture lookups in total. I tried to 
// make it as concise as possible. Whether that translates to speed, or not, I couldn't say.
float3 texBump( SamplerState tx, in float3 p, in float3 n, float bf){
   
    const float2 e = float2(0.001, 0);
    
    // Three gradient vectors rolled into a matrix, constructed with offset greyscale texture values.    
    float3x3 m = float3x3( tex3D(tx, p - e.xyy, n), tex3D(tx, p - e.yxy, n), tex3D(tx, p - e.yyx, n));
    
    float3 g = mul(float3(0.299, 0.587, 0.114),m); // Converting to greyscale.
    g = (g - dot(tex3D(tx,  p , n), float3(0.299, 0.587, 0.114)) )/e.x; g -= n*dot(n, g);
                      
    return normalize( n + g*bf ); // Bumped normal. "bf" - bump factor.
}

float4 psFractalFlythrough(VS_OUT In):SV_Target{
	// Screen coordinates.
    float2 q = 1-In.TexCd.rg;
	float2 u = -1.0 + 2.0*q;
    
	float speed = iTime*0.35 + 8.;
    
    // Initiate the camera path spline points. Kind of wasteful not making this global, but I wanted
    // it self contained... for better or worse. I'm not really sure what the GPU would prefer.
    setCamPath();

    
	// Camera Setup.
    float3 ro = camPath(speed); // Camera position, doubling as the ray origin.
    float3 lk = camPath(speed + .5);  // "Look At" position.
    float3 lp = camPath(speed + .5) + float3(0, .25, 0); // Light position, somewhere near the moving camera.
   
    
    // Using the above to produce the unit ray-direction vector.
    float FOV = 1.57; // FOV - Field of view.
    float3 fwd = normalize(lk-ro);
    float3 rgt = normalize(float3(fwd.z, 0, -fwd.x));
    float3 up = (cross(fwd, rgt));
    
	// Unit direction ray.
    float3 rd = normalize(In.PosW.xyz);
    
    // Raymarch the scene.
    float t = trace(ro, rd);
    
    // Initialize the scene color.
    float3 col = float3(0,0,0);
    
    // Scene hit, so color the pixel. Technically, the object should always be hit, so it's tempting to
    // remove this entire branch... but I'll leave it, for now.
    if(t<FAR){
        
        // This looks a little messy and haphazard, but it's really just some basic lighting, and application
        // of the following material properties: Wood = 1., Metal = 2., Gold = 3..
    
        float ts = 1.;  // Texture scale.
        
        // Global object ID. It needs to be saved just after the raymarching equation, since other "map" calls,
        // like normal calculations will give incorrect results. Found that out the hard way. :)
        float saveObjID = objID; 
        
        
        float3 pos = ro + rd*t; // Scene postion.
        float3 nor = calcNormal(pos); // Normal.
        float3 sNor = nor;
        
        
        // Apply some subtle texture bump mapping to the panels and the metal tubing.
        nor = texBump(s0, pos*ts, nor, 0.002); // + step(saveObjID, 1.5)*0.002
    
        // Reflected ray. Note that the normal is only half bumped. It's fake, but it helps
        // taking some of the warping effect off of the reflections.
        float3 ref = reflect(rd, normalize(sNor*.5 + nor*.5)); 
         
        
		col = tex3D(s0, pos*ts, nor); // Texture pixel at the scene postion.
        
        
        float3  li = lp - pos; // Point light.
        float lDist = max(length(li), .001); // Surface to light distance.
        float atten = 1./(1.0 + lDist*0.125 + lDist*lDist*.05); // Light attenuation.
        li /= lDist; // Normalizing the point light vector.
        
        float occ = calcAO( pos, nor ); // Occlusion.
		
        float dif = clamp(dot(nor, li), 0.0, 1.0); // Diffuse.
        dif = pow(dif, 4.)*2.;
        float spe = pow(max(dot(reflect(-li, nor), -rd), 0.), 8.); // Object specular.
        float spe2 = spe*spe; // Global specular.
        
        float refl = .35; // Reflection coefficient. Different for different materials.

            

        // Reflection color. Mostly fake.
        // Cheap reflection: Not entirely accurate, but the reflections are pretty subtle, so not much 
        // effort is being put in.
        float rt = refTrace(pos + ref*0.1, ref); // Raymarch from "sp" in the reflected direction.
        float rSaveObjID = objID; // IDs change with reflection. Learned that the hard way. :)
        float3 rsp = pos + ref*rt; // Reflected surface hit point.
        float3 rsn = calcNormal(rsp); // Normal at the reflected surface. Too costly to bump reflections.
        float3 rCol = tex3D(s0, rsp*ts, rsn); // Texel at "rsp."
        float3 rLi = lp-rsp;
        float rlDist = max(length(rLi), 0.001);
        rLi /= rlDist;
        float rDiff = max(dot(rsn, rLi), 0.); // Diffuse light at "rsp."
        rDiff = pow(rDiff, 4.)*2.;
        float rAtten = 1./(1. + rlDist*0.125 + rlDist*rlDist*.05);
        
        if(rSaveObjID>1.5 && rSaveObjID<2.5){
            rCol = float3(1,1,1)*dot(rCol, float3(.299, .587, .114))*.7 + rCol*.15;//*.7+.2
            //rDiff *= 1.35;
        }
        if(rSaveObjID>2.5){
             //float rc = dot(rCol, vec3(.299, .587, .114));
             float3 rFire = pow(float3(1.5, 1, 1)*rCol, float3(8, 2, 1.5));//*.5+rc*.5;
             rCol = min(lerp(float3(1.5, .9, .375), float3(.75, .375, .3), rFire), 2.)*.5 + rCol;         
        }
        
        rCol *= (rDiff + .35)*rAtten; // Reflected color. Not accurate, but close enough.         
        
        
        
        // Grey metal inner tubing.
        if(saveObjID>1.5 && saveObjID<2.5){ 
			
            // Grey out the limestone wall color.
            col = float3(1,1,1)*dot(col, float3(.299, .587, .114))*.7 + col*.15;
            
            refl = .5;
            //dif *= 1.35;
            //spe2 *= 1.35;
            
        }         
        
        // Gold trimming properties. More effort should probably be put in here.
        // I could just write "saveObjID == 3.," but I get a little paranoid where floats are concerned. :)
        if(saveObjID>2.5){

            // For the screen image, we're interested in the offset height and depth positions. Ie: pOffs.zy.
            
            // Pixelized dot pattern shade.
            //float c = dot(col, vec3(.299, .587, .114));
            
            float3 fire = pow(float3(1.5, 1, 1)*col, float3(8, 2, 1.5));//*.5+c*.5;
            col = min(lerp(float3(1, .9, .375), float3(.75, .375, .3), fire), 2.)*.5 + col;//
            
            refl = .65;
            //dif *= 1.5;
            //spe2 *= 1.5;
            
        }
        
     
        // Combining everything together to produce the scene color.
        col = col*(dif + .35  + float3(.35, .45, .5)*spe) + float3(.7, .9, 1)*spe2 + rCol*refl;
        col *= occ*atten; // Applying occlusion.
       
        
    }

    
    // Applying some very slight fog in the distance. This is technically an inside scene...
    // Or is it underground... Who cares, it's just a shader. :)
    col = lerp(min(col, 1.), float3(0,0,0), 1.-exp(-t*t/FAR/FAR*20.));//smoothstep(0., FAR-20., t)
    //col = mix(min(col, 1.), vec3(0), smoothstep(0., FAR-35., t));//smoothstep(0., FAR-20., t)
    
    
    
    // Done.
    return  float4(sqrt(max(col, 0.)), 1.0);
}


technique10 FractalFlythrough{
	pass P0{
		SetVertexShader(CompileShader(vs_4_0,VS()));
		SetPixelShader(CompileShader(ps_4_0,psFractalFlythrough()));
	}
}



