//@author: umanema
//@help: a skybox implementation of a shader from shadertoy
//@tags: skybox, raymarching, shadertoy
//@credits: David Hoskins
//@license: Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

static const float3 sunColour = float3(1.0, .95, .8);

#define SCALE 2.8
#define MINRAD2 .25
static const float minRad2 = clamp(MINRAD2, 1.0e-9, 1.0);
#define scale (float4(SCALE, SCALE, SCALE, abs(SCALE)) / minRad2)
static const float absScalem1 = abs(SCALE - 1.0);
static const float AbsScaleRaisedTo1mIters = pow(abs(SCALE), float(1-10));

cbuffer cbControls:register(b0){
	float4x4 tW:WORLD;
	float4x4 tV:VIEW;
	float4x4 tP:PROJECTION;
	float4x4 tVI:VIEWINVERSE;
	float4x4 tPI:PROJECTIONINVERSE;
	float4x4 tWIT:WORLDINVERSETRANSPOSE;
	
	float3 sunDir  = normalize( float3(  0.35, 0.1,  0.3 ) );
	float3 surfaceColour1 = float3(.8, .0, 0.);
	float3 surfaceColour2 = float3(.4, .4, 0.5);
	float3 surfaceColour3 = float3(.5, 0.3, 0.00);
	float3 fogCol = float3(0.4, 0.4, 0.4);
	float iTime;
	float2 TargetSize = {512,512};
	
};

struct VS_IN{
	float4 PosO:POSITION;
	float4 TexCd:TEXCOORD0;
};

struct VS_OUT{
	float4 PosWVP:SV_POSITION;
	float4 TexCd : TEXCOORD0;
	float4 PosW:TEXCOORD1;
};

VS_OUT VS(VS_IN In){
	VS_OUT Out=(VS_OUT)0;
	float4 PosW=In.PosO;
	PosW=mul(PosW,tW);
	Out.PosW=PosW;
	Out.TexCd = In.TexCd;
	Out.PosWVP=mul(PosW,mul(tV,tP));
	return Out;
}

//------------------------------------------------------------------------------
//use procedural noise instead of original
//------------------------------------------------------------------------------
float hash( float n )
{
	return frac(sin(n)*43758.5453);
}
float Noise( in float3 x )
{
	float3 p = floor(x);
	float3 f = frac(x);
	
	f = f*f*(3.0-2.0*f);
	float n = p.x + p.y*57.0 + 113.0*p.z;
	return lerp(lerp(lerp( hash(n+  0.0), hash(n+  1.0),f.x),
	lerp( hash(n+ 57.0), hash(n+ 58.0),f.x),f.y),
	lerp(lerp( hash(n+113.0), hash(n+114.0),f.x),
	lerp( hash(n+170.0), hash(n+171.0),f.x),f.y),f.z);
}

//------------------------------------------------------------------------------
float Map(float3 pos) 
{
	//return (length(pos)-4.0);

	float4 p = float4(pos,1);
	float4 p0 = p;  // p.w is the distance estimate

	for (int i = 0; i < 9; i++)
	{
		p.xyz = clamp(p.xyz, -1.0, 1.0) * 2.0 - p.xyz;

		// sphere folding: if (r2 < minRad2) p /= minRad2; else if (r2 < 1.0) p /= r2;
		float r2 = dot(p.xyz, p.xyz);
		p *= clamp(max(minRad2/r2, minRad2), 0.0, 1.0);

		// scale, translate
		p = p*scale + p0;
	}
	return ((length(p.xyz) - absScalem1) / p.w - AbsScaleRaisedTo1mIters);
}

//----------------------------------------------------------------------------------------
float3 Colour(float3 pos, float sphereR) 
{
	//scale.z = 0.0;
	float3 p = pos;
	float3 p0 = p;
	float trap = 1.0;
    
	for (int i = 0; i < 6; i++)
	{
        
		p.xyz = clamp(p.xyz, -1.0, 1.0) * 2.0 - p.xyz;
		float r2 = dot(p.xyz, p.xyz);
		p *= clamp(max(minRad2/r2, minRad2), 0.0, 1.0);

		p = p*scale.xyz + p0.xyz;
		trap = min(trap, r2);
	}
	// |c.x|: log final distance (fractional iteration count)
	// |c.y|: spherical orbit trap at (0,0,0)
	float2 c = clamp(float2( 0.3333*log(dot(p,p))-1.0, sqrt(trap) ), 0.0, 1.0);

    float t = fmod(length(pos) + iTime*150.0, 16.0);
    float3 sC1 = lerp( surfaceColour1, float3(.4, 3.0, 5.),
					 pow(smoothstep(0.0, .3, t) * smoothstep(0.6, .3, t), 10.0));
	return lerp(lerp(sC1, surfaceColour2, c.y), surfaceColour3, c.x);
}


//------------------------------------------------------------------------------
float3 GetNormal(float3 pos, float distance)
{
    distance *= 0.001+.0001;
	float2 eps = float2(distance, 0.0);
	float3 nor = float3(
	    Map(pos+eps.xyy) - Map(pos-eps.xyy),
	    Map(pos+eps.yxy) - Map(pos-eps.yxy),
	    Map(pos+eps.yyx) - Map(pos-eps.yyx));
	return normalize(nor);
}

//----------------------------------------------------------------------------------------
float GetSky(float3 pos)
{
    pos *= 2.3;
	float t = Noise(pos);
    t += Noise(pos * 2.1) * .5;
    t += Noise(pos * 4.3) * .25;
    t += Noise(pos * 7.9) * .125;
	return t;
}

//----------------------------------------------------------------------------------------
float BinarySubdivision(in float3 rO, in float3 rD, float2 t)
{
    float halfwayT;
  
    for (int i = 0; i < 6; i++)
    {

        halfwayT = dot(t, float2(.5,.5));
        float d = Map(rO + halfwayT*rD); 
        //if (abs(d) < 0.001) break;
        t = lerp(float2(t.x, halfwayT), float2(halfwayT, t.y), step(0.0005, d));

    }

	return halfwayT;
}

//----------------------------------------------------------------------------------------
float2 Scene(in float3 rO, in float3 rD, in float2 fragCoord)
{
	float t = .05 + 0.05 * Noise(rO);
	float3 p = float3(0.0,0.0,0.0);
    float oldT = 0.0;
    bool hit = false;
    float glow = 0.0;
    float2 dist;
	for( int j=0; j < 100; j++ )
	{
		if (t > 12.0) break;
        p = rO + t*rD;
       
		float h = Map(p);
        
		if(h  <0.0005)
		{
            dist = float2(oldT, t);
            hit = true;
            break;
        }
       	glow += clamp(.05-h, 0.0, .4);
        oldT = t;
      	t +=  h + t*0.001;
 	}
    if (!hit)
        t = 1000.0;
    else t = BinarySubdivision(rO, rD, dist);
    return float2(t, clamp(glow*.25, 0.0, 1.0));

}

//----------------------------------------------------------------------------------------
float Hash(float2 p)
{
	return frac(sin(dot(p, float2(12.9898, 78.233))) * 33758.5453)-.5;
} 

//----------------------------------------------------------------------------------------
float3 PostEffects(float3 rgb, float2 xy)
{
	// Gamma first...
	

	// Then...
	#define CONTRAST 1.08
	#define SATURATION 1.5
	#define BRIGHTNESS 1.5
	rgb = lerp(float3(0.5,0.5,0.5), lerp(float(dot(float3(.2125, .7154, .0721), rgb*BRIGHTNESS)), rgb*BRIGHTNESS, SATURATION), CONTRAST);
	// Noise...
	//rgb = clamp(rgb+Hash(xy*iTime)*.01, 0.0, 1.0);

	rgb = pow(rgb, float3(0.47,0.47,0.47));
	return rgb;
}

//----------------------------------------------------------------------------------------
float Shadow( in float3 ro, in float3 rd)
{
	float res = 1.0;
    float t = 0.05;
	float h;
	
    for (int i = 0; i < 8; i++)
	{
		h = Map( ro + rd*t );
		res = min(6.0*h / t, res);
		t += h;
	}
    return max(res, 0.0);
}

//----------------------------------------------------------------------------------------
float3x3 RotationMatrix(float3 axis, float angle)
{
    axis = normalize(axis);
    float s = sin(angle);
    float c = cos(angle);
    float oc = 1.0 - c;
    
    return float3x3(oc * axis.x * axis.x + c, oc * axis.x * axis.y - axis.z * s,  oc * axis.z * axis.x + axis.y * s,
                oc * axis.x * axis.y + axis.z * s,  oc * axis.y * axis.y + c, oc * axis.y * axis.z - axis.x * s,
                oc * axis.z * axis.x - axis.y * s,  oc * axis.y * axis.z + axis.x * s,  oc * axis.z * axis.z + c);
}

//----------------------------------------------------------------------------------------
float3 LightSource(float3 spotLight, float3 dir, float dis)
{
    float g = 0.0;
    if (length(spotLight) < dis)
    {
		g = pow(max(dot(normalize(spotLight), dir), 0.0), 500.0);
    }
	
    return float3(0.6,0.6,0.6) * g;
}

//----------------------------------------------------------------------------------------
float3 CameraPath( float t )
{
    float3 p = float3(-.78 + 3. * sin(2.14*t),.05+2.5 * sin(.942*t+1.3),.05 + 3.5 * cos(3.594*t) );
	return p;
}

float4 psRemnantX(VS_OUT In): SV_Target {
	
	float2 q = 1-In.TexCd.rg;
	float2 p = -1.0 + 2.0*q;
	
	float3 cameraPos = CameraPath(iTime);
    float3 camTar = CameraPath(iTime + .01);
	
	
	float roll = 13.0*sin(iTime*.5+.4);
	float3 cw = normalize(camTar-cameraPos);

	float3 cp = float3(sin(roll), cos(roll),0.0);
	float3 cu = normalize(cross(cw,cp));

	float3 cv = normalize(cross(cu,cw));
    cw = mul(RotationMatrix(cv, sin(-iTime*20.0)*.7), cw);
	//float3 dir = normalize(p.x*cu + p.y*cv + 1.3*cw);
	float3 dir = normalize(In.PosW.xyz);
	
    float3 spotLight = CameraPath(iTime + .03) + float3(sin(iTime*18.4), cos(iTime*17.98), sin(iTime * 22.53))*.2;
	float3 col = float3(0.0,0.0,0.0);
    float3 sky = float3(0.03, .04, .05) * GetSky(dir);
	float2 ret = Scene(cameraPos, dir, In.PosW.xy);
    
	if (ret.x < 900.0)
	{
		float3 p = cameraPos + ret.x*dir; 
		float3 nor = GetNormal(p, ret.x);
        
		float3 spot = spotLight - p;
		float atten = length(spot);

		spot /= atten;
        
		float shaSpot = Shadow(p, spot);
		float shaSun = Shadow(p, sunDir);
        
		float bri = max(dot(spot, nor), 0.0) / pow(atten, 1.5) * .15;
		float briSun = max(dot(sunDir, nor), 0.0) * .3;
        
		col = Colour(p, ret.x);
		col = (col * bri * shaSpot) + (col * briSun* shaSun);
		
		float3 ref = reflect(dir, nor);
		col += pow(max(dot(spot,   ref), 0.0), 10.0) * 2.0 * shaSpot * bri;
		col += pow(max(dot(sunDir, ref), 0.0), 10.0) * 2.0 * shaSun  * bri;
    }
    
    col = lerp(sky, col, min(exp(-ret.x+1.5), 1.0));
    col += float3(pow(abs(ret.y), 2.),pow(abs(ret.y), 2.),pow(abs(ret.y), 2.)) * float3(.02, .04, .1);

    col += LightSource(spotLight-cameraPos, dir, ret.x);
	col = PostEffects(col, q);	
	
	return float4(col,1.0);
}


technique10 RemnantX{
	pass P0{
		SetVertexShader(CompileShader(vs_4_0,VS()));
		SetPixelShader(CompileShader(ps_4_0,psRemnantX()));
	}
}



