//@author: umanema
//@help: a skybox implementation of a shader from shadertoy
//@tags: skybox, raymarching, shadertoy
//@credits: zackpudil
//@license: Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

cbuffer cbControls:register(b0){
	float4x4 tW:WORLD;
	float4x4 tV:VIEW;
	float4x4 tP:PROJECTION;
	float4x4 tVI:VIEWINVERSE;
	float4x4 tPI:PROJECTIONINVERSE;
	float4x4 tWIT:WORLDINVERSETRANSPOSE;
	
	float iTime = 0.0;
	
};

struct VS_IN{
	float4 PosO:POSITION;
	float4 TexCd:TEXCOORD0;
};

struct VS_OUT{
	float4 PosWVP:SV_POSITION;
	float4 TexCd:TEXCOORD0;
	float4 PosW:TEXCOORD1;
};

VS_OUT VS(VS_IN In){
	VS_OUT Out=(VS_OUT)0;
	float4 PosW=In.PosO;
	PosW=mul(PosW,tW);
	Out.TexCd = In.TexCd;
	Out.PosW=PosW;
	Out.PosWVP=mul(PosW,mul(tV,tP));
	return Out;
}

static const float tmax = 20.0;

float hash(float n) {
	return frac(sin(n)*43758.5453);
}

float3 hash(float3 p){ 
    
    float n = sin(dot(p, float3(7, 157, 113)));    
    return frac(float3(2097152, 262144, 32768)*n); 
}

float noise(float g) {
	float p = floor(g);
	float f = frac(g);

	return lerp(hash(p), hash(p + 1.0), f);
}

float voronoi(float3 x) {
	float3 p = floor(x);
	float3 f = frac(x);

	float2 res = float2(8.0,8.0);

	for(int i = -1; i <= 1; i++)
	for(int j = -1; j <= 1; j++)
	for(int k = -1; k <= 1; k++) {
		float3 g = float3(float(i), float(j), float(k));
		float3 r = g + hash(p + g) - f;

		float d = max(abs(r.x), max(abs(r.y), abs(r.z)));

		if(d < res.x) {
			res.y = res.x;
			res.x = d;
		} else if(d < res.y) {
			res.y = d;
		}
	}

	return res.y - res.x;
}

float2 path(float z) {
	return float2(cos(z/8.0)*sin(z/12.0)*12.0, 0.0);
}

float map(float3 p) {
	float4 q = float4(p, 1.0);
	q.x += 1.0;

	for(int i = 0; i < 6; i++) {
		q.xyz = -1.0 + 2.0*frac(0.5 + 0.5*q.xyz);
		q = 1.2*q/max(dot(q.xyz, q.xyz), 0.1);
	}

	float2 tun = abs(p.xy - path(p.z))*float2(0.6, 0.5);

	return min(0.25*abs(q.y)/q.w, 1.0 - max(tun.x, tun.y));
}

float march(float3 ro, float3 rd, float mx) {
	float t = 0.0;

	for(int i = 0; i < 200; i++) {
		float d = map(ro + rd*t);
		if(d < 0.001 || t >= mx) break;
		t += d*0.75;
	}

	return t;
}

float3 normal(float3 p) {
	float2 h = float2(0.001, 0.0);
	float3 n = float3(
		map(p + h.xyy) - map(p - h.xyy),
		map(p + h.yxy) - map(p - h.yxy),
		map(p + h.yyx) - map(p - h.yyx)
	);
	return normalize(n);
}

float ao(float3 p, float3 n) {
    float o = 0.0, s = 0.005;
    for(int i = 0; i< 15; i++) {
        float d = map(p + n*s);
        o += (s - d);
        s += s/float(i + 1);
    }
    
    return 1.0 - clamp(o, 0.0, 1.0);
}

float3 material(float3 p) {
	float v = 0.0;
	float a = 0.7, f = 1.0;

	for(int i = 0; i < 4; i++) {
		float v1 = voronoi(p*f + 5.0);
		float v2 = 0.0;

		if(i > 0) {
			v2 = voronoi(p*f*0.1 + 50.0 + 0.15*iTime);

			float va = 0.0, vb = 0.0;
			va = 1.0 - smoothstep(0.0, 0.1, v1);
			vb = 1.0 - smoothstep(0.0, 0.08, v2);
			v += a*pow(va*(0.5 + vb), 4.0);
		}

		v1 = 1.0 - smoothstep(0.0, 0.3, v1);
		v2 =  a*noise(v1*5.5 + 0.1);

		v += v2;

		f *= 3.0;
		a *= 0.5;
	}

	return float3(pow(v, 6.0), pow(v, 4.0), pow(v, 2.0))*2.0;
}

float3x3 camera(float3 eye, float3 lat) {
	float3 ww = normalize(lat - eye);
	float3 uu = normalize(cross(float3(0, 1, 0), ww));
	float3 vv = normalize(cross(ww, uu));

	return float3x3(uu, vv, ww);
}

float4 psServerRoom(VS_OUT In):SV_Target {
	
	float3 col = float3(0,0,0);

	float3 ro = float3(0.63*cos(iTime*0.1), 0.67, iTime*0.5);
	float3 la = ro + float3(0, 0, 0.3);

	ro.xy += path(ro.z);
	la.xy += path(la.z);
	float3 rd = normalize(In.PosW.xyz);
	
	float i = march(ro, rd, tmax);
	if(i < tmax) {
		float3 pos = ro + rd*i;
		float3 nor = normal(pos);
		float3 ref = normalize(reflect(rd, nor));
        
        float occ = ao(pos, nor);
        
		col = material(pos)*occ;
        col += 2.0*pow(clamp(dot(-rd, ref), 0.0, 1.0), 16.0)*occ;
	}

	col = lerp(col, float3(0,0,0), 1.0 - exp(-0.5*i));
    
    col = 1.0 - exp(-0.5*col);
    col = pow(abs(col), float3(1.0/2.2,1.0/2.2,1.0/2.2));

	return float4(col, 1);
}

technique10 ServerRoom{
	pass P0{
		SetVertexShader(CompileShader(vs_4_0,VS()));
		SetPixelShader(CompileShader(ps_4_0,psServerRoom()));
	}
}


