//@author: antokhio
//@description: Lava Effect by Sergey A. Makovkin (sergeymak@pisem.net)
//@tags: lava fire volumetric animated
//@credits: nVidia dottore

/******************************************************************************

Copyright NVIDIA Corporation 2004
TO THE MAXIMUM EXTENT PERMITTED BY APPLICABLE LAW, THIS SOFTWARE IS PROVIDED
*AS IS* AND NVIDIA AND ITS SUPPLIERS DISCLAIM ALL WARRANTIES, EITHER EXPRESS
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL NVIDIA OR ITS SUPPLIERS
BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT, OR CONSEQUENTIAL DAMAGES
WHATSOEVER (INCLUDING, WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR ANY OTHER PECUNIARY LOSS)
ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF NVIDIA HAS
BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

******************************************************************************/

#define MARBLE_SIZE 64

#define MARBLE_COL_PALE float3(0.25, 0.25, 0.35)
#define MARBLE_COL_MED float3(0.40, 0.30, 0.30)
#define MARBLE_COL_DARK float3(0.05, 0.05, 0.26)
#define MARBLE_COL_DARKER float3(0.03, 0.03, 0.20)

#define MARBLE_KNOTS 13
#define CR00	-0.5
#define CR01	 1.5
#define CR02	-1.5
#define CR03	 0.5
#define CR10	 1.0
#define CR11	-2.5
#define CR12	 2.0
#define CR13	-0.5
#define CR20	-0.5
#define CR21	 0.0
#define CR22	 0.5
#define CR23     0.0
#define CR30     0.0
#define CR31	 1.0
#define CR32     0.0
#define CR33     0.0

float3 InterpSplineCatmullRom(float u, float3 knot0, float3 knot1, float3 knot2, float3 knot3 )
{
	float3 c0;
	float3 c1;
	float3 c2;
	float3 c3;

	c3 = (CR00 * knot0) + (CR01 * knot1) + (CR02 * knot2) + (CR03 * knot3);
	c2 = (CR10 * knot0) + (CR11 * knot1) + (CR12 * knot2) + (CR13 * knot3);
	c1 = (CR20 * knot0) + (CR21 * knot1) + (CR22 * knot2) + (CR23 * knot3);
	c0 = (CR30 * knot0) + (CR31 * knot1) + (CR32 * knot2) + (CR33 * knot3);

	return ((c3*u + c2)*u + c1)*u + c0;
}

static const float3 marble_knots[MARBLE_KNOTS+4] = 
{
  MARBLE_COL_PALE,  // dummy
  MARBLE_COL_PALE,  // dummy
  MARBLE_COL_PALE,      // 0
  MARBLE_COL_PALE,
  MARBLE_COL_MED, 
  MARBLE_COL_MED, 
  MARBLE_COL_MED,
  MARBLE_COL_PALE, 
  MARBLE_COL_PALE,
  MARBLE_COL_DARK, 
  MARBLE_COL_DARK,
  MARBLE_COL_DARKER, 
  MARBLE_COL_DARKER,
  MARBLE_COL_PALE, 
  MARBLE_COL_DARKER,     // 1
  MARBLE_COL_DARKER, // dummy
  MARBLE_COL_DARKER, // dummy
};

// put map in texture 4, 64 entries cubic interpolation over the surface
float3 make_marble_map(float x)
{
    float3 knot0, knot1, knot2, knot3;

    // need to get 0-1 -> 1-13 for knot values
    float f = x * 15.0 + 1.0;

    float k = floor(f); // get lower integer for knots

    knot0 = marble_knots[k - 1];
    knot1 = marble_knots[k];
    knot2 = marble_knots[k + 1];
    knot3 = marble_knots[k + 2];
    
    return InterpSplineCatmullRom(f - k, knot0, knot1, knot2, knot3);            
}

float4 GenerateVolumeMap(float3 p)
{
	return (noise(p * 50.5) * .5) + .5f;
}

float3 MarbleColor(float3 p )
{
	return make_marble_map(p.x);
}

Texture2D BarkMap;
SamplerState s0: IMMUTABLE {
	Filter=MIN_MAG_MIP_LINEAR;
	AddressU=WRAP;
	AddressV=WRAP;
};

Texture3D VolumeMap;
SamplerState s1:IMMUTABLE {
	Filter=MIN_MAG_MIP_LINEAR;
	AddressU=WRAP;
	AddressV=WRAP;
	AddressW=WRAP;
};

float4x4 tW : WORLD;
float4x4 tVP : VIEWPROJECTION;
float4x4 tWIT : WORLDINVERSETRANSPOSE;
float4x4 tWVP : WORLDVIEWPROJEWCTION;
float4x4 tVI: VIEWINVERSE;
float4x4 tTex <bool uvspace=true; string uiname="Texture Transform";>;

float Time = 0.0;
float4 LavaColor1<string uiname="Lava Color 1"; bool color = true;> = {0.8, 0.8, 0.8, 1.0};
float4 LavaColor2<string uiname="Lava Color 2"; bool color = true;> = {0.8, 0.8, 0.8, 1.0};
float LavaFactor<string uiname="Lava Factor"; float uimin = 0.01; float uimax = 1.00;> = 0.1;


struct VS_IN
{
	float4 pos : POSITION;
	float4 normal: NORMAL;
	float4 uv : TEXCOORD0;
};

struct VS_OUT
{
    float4 pos: SV_POSITION;
    float4 uv: TEXCOORD0;
	float3 normal: NORMAL;
	float3 posW: TEXCOORD1;
	float3 camera: TEXCOORD2;
};

VS_OUT VS(VS_IN input)
{
    VS_OUT output = (VS_OUT)0;
    output.pos  = mul(input.pos,mul(tW,tVP));
    output.uv = mul(input.uv, tTex);
	output.normal = mul(input.normal, tWIT).xyz;
	output.posW = mul(input.pos, tW).xyz;
	output.camera = tVI[3].xyz; 
    return output;
}

float4 PS(VS_OUT input): SV_Target
{
    float rnd = 0.0f;
	float f = 1;
	float3 uvw = input.posW + (Time * .03f);
	for (int i = 0; i < 4; i++)
	{
		half4 fnoise = VolumeMap.SampleLevel(s1, uvw * 0.2 * f, 0);
		fnoise -=.5f;
		fnoise *= 4.0f;
		rnd += ( fnoise.x) / f;
		f *= 4.17;	
	}
	    
	float3 coord = input.posW;
	coord.x += rnd*LavaFactor;
	coord.y += rnd*LavaFactor;
	float4 tex = BarkMap.SampleLevel(s0, coord.xy, 0);
	
	
	return tex * LavaColor1 * (rnd + 0.1) * 10 + LavaColor2;
}

technique10 Template
{
	pass P0
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );
		SetPixelShader( CompileShader( ps_4_0, PS() ) );
	}
}




