﻿#region usings

using System;
using System.Linq;
using DAP.CompGeom;
using VVVV.PluginInterfaces.V2;

#endregion usings

namespace VVVV.Nodes
{
    #region PluginInfo
	[PluginInfo(Name = "CellVertices", Category = "2d.Voronoi", Help = "Returns voronoi vertices as 2D points in CCW order", Tags = "Voronoi, Delaunay")]
    #endregion PluginInfo
    public class VoronoiCellVertices : IPluginEvaluate
    {
		public enum InsetType
		{
			Percentage,
			Absolute
		}
        #region fields & pins
        [Input("Cell")]
		public IDiffSpread<FortunePoly> Cell;

		[Input("Inset", DefaultValue = 1.0, MinValue = 0.0)]
		public IDiffSpread<double> Inset;

		[Input("Infinite Ray Multiplier", IsBang = true, DefaultValue = 100.0)]
		public IDiffSpread<double> RayMult;

		[Input("Inset Type")]
		public IDiffSpread<InsetType> InsetTypeSprd;

		[Output("Points")]
		public ISpread<double> Points;

        [Output("Bin Size")]
		public ISpread<int> BinSize;

	    [Output("Center")]
		public ISpread<double> Center; 
        #endregion fields & pins


        //called when data for any output pin is requested
        public void Evaluate(int spreadMax)
        {
	        if (!Cell.IsChanged && !Inset.IsChanged && !RayMult.IsChanged && !InsetTypeSprd.IsChanged)
	        {
		        return;
	        }
            Points.SliceCount = 0;
	        Center.SliceCount = 2 * Cell.SliceCount;
			BinSize.SliceCount = Cell.SliceCount;
            var rayMult = RayMult[0];
            for (var icell = 0; icell < Cell.SliceCount; icell++)
            {
                var cell = Cell[icell];
                if (cell == null || cell.FAtInfinity)
                {
                    continue;
                }
	            var inset = Inset[icell];
                var binSize = 0;
	            if (InsetTypeSprd[0] == InsetType.Percentage)
	            {
					foreach (var realVertex in cell.RealVertices(rayMult))
					{
						var newVertex = realVertex;
						// ReSharper disable once CompareOfFloatsByEqualityOperator
						if (inset != 0.0)
						{
							newVertex.X = cell.VoronoiPoint.X + (1 - inset) * (newVertex.X - cell.VoronoiPoint.X);
							newVertex.Y = cell.VoronoiPoint.Y + (1 - inset) * (newVertex.Y - cell.VoronoiPoint.Y);
						}
						Points.Add(newVertex.X);
						Points.Add(newVertex.Y);
						binSize++;
					}
	            }
	            else
	            {
					var vertexArray = cell.RealVertices(rayMult).ToList();
					var cvtx = vertexArray.Count;
					binSize = cvtx;

					for (var ivtx = 0; ivtx < cvtx; ivtx++)
					{
						// ReSharper disable once CompareOfFloatsByEqualityOperator
						if (inset != 0.0)
						{
							var prevVtx = vertexArray[(ivtx + cvtx - 1) % cvtx];
							var nextVtx = vertexArray[(ivtx + 1) % cvtx];
							var newVertex = vertexArray[ivtx];
							var vctNext = NormalVectorTo(newVertex, nextVtx);
							var vctPrev = NormalVectorTo(newVertex, prevVtx);
							var bisectorVector = vctPrev + vctNext;
							var bisectorPt = bisectorVector + newVertex;
							var bisectorDistance = PointToLineDistance(bisectorPt, prevVtx, newVertex);
							var ratio = inset / bisectorDistance;
							Points.Add(newVertex.X + ratio * bisectorVector.X);
							Points.Add(newVertex.Y + ratio * bisectorVector.Y);
						}
						else
						{
							Points.Add(vertexArray[ivtx].X);
							Points.Add(vertexArray[ivtx].Y);
						}
					}
	            }
                BinSize[icell] = binSize;
	            Center[2 * icell] = cell.VoronoiPoint.X;
	            Center[2 * icell + 1] = cell.VoronoiPoint.Y;
            }
        }
		
		static PointD NormalVectorTo(PointD from, PointD to)
		{
			var diff = to - from;
			diff.Normalize();
			return diff;
		}

		static double PointToLineDistance(PointD pt, PointD pt1, PointD pt2)
		{
			var dx21 = pt2.X - pt1.X;
			var dy21 = pt2.Y - pt1.Y;
			var num = Math.Abs(dx21 * (pt1.Y - pt.Y) - dy21 * (pt1.X - pt.X));
			var den = Math.Sqrt(dx21 * dx21 + dy21 * dy21);
			// ReSharper disable once CompareOfFloatsByEqualityOperator
			return den == 0 ? 
				Math.Sign(num) * Math.Sign(den) < 0 ? double.MinValue : double.MaxValue :
				num / den;
		}
    }
}
