﻿#region usings
using System;
using System.ComponentModel.Composition;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Diagnostics;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;

using VVVV.Core.Logging;

//youtube - google stuff
using Google.GData.Client;
using Google.GData.Client.ResumableUpload;
using Google.GData.Extensions;
using Google.GData.YouTube;
using Google.GData.Extensions.MediaRss;
using Google.YouTube;

#endregion usings

namespace VVVV.Nodes
{
    #region PluginInfo
    [PluginInfo(
        Name = "YouTube", 
        Category = "Network", 
        Help = "Uploads videos to YouTube Channel",
        Author = "ethermammoth",
        Tags = "")]
    #endregion PluginInfo
    public class NetworkYouTubeNode : IPluginEvaluate
    {
        #region fields & pins
        [Input("Video Title", DefaultString = "VVVV", IsSingle = true)]
        ISpread<string> FVideoTitle;

        [Input("Video Description", DefaultString = "VVVV", IsSingle = true)]
        ISpread<string> FVideoDescription;

        [Input("Video Keywords", DefaultString = "vvvv, fun", IsSingle = true)]
        ISpread<string> FVideoKeywords;

        [Input("Video Category", EnumName = "YouTubeTags")]
        IDiffSpread<EnumEntry> FVideoTags;

        [Input("Video Private", IsSingle = true)]
        ISpread<bool> FVideoPrivate;

        [Input("UserName", DefaultString = "mail@host.com", IsSingle = true)]
        ISpread<string> FUserName;

        [Input("Password", DefaultString = "mypassword", IsSingle = true)]
        ISpread<string> FPassword;

        [Input("Link to Video", DefaultString = "video.mp4", IsSingle = true)]
        ISpread<string> FVideoLink;

        [Input("Send", IsBang = true, IsSingle = true)]
        IDiffSpread<bool> FPinInDoSend;

        [Input("Application Name", DefaultString = "vvvvv", IsSingle = true, Visibility = PinVisibility.Hidden)]
        ISpread<string> FApplicationName;

        [Input("Developer Key", DefaultString = "AI39si4z_Tw-YB70TaS7Q212SFWG4V8v7JIb0uWnPZSoE2RXl8M6mnwOKjB_8abDmU3bJBb2zChxZct6QobTXvwPKFbyH4zWsQ", IsSingle = true, Visibility = PinVisibility.Hidden)]
        ISpread<string> FDevKey;

        [Output("Progress")]
        ISpread<int> FRequestProgress;

        [Output("VideoInformation Response")]
        ISpread<string> FVideoInformationResponse;

        [Output("YouTube URL")]
        ISpread<string> FResponse;

        [Output("Finished Uploading")]
        ISpread<bool> FFinishedUploading;

        [Import()]
        ILogger FLogger;

        [ImportingConstructor]
        public NetworkYouTubeNode()
		{
            var s = new string[] { "Film", "Autos", "Music", "Animals", "Sports", "Shortmov", 
            "Travel", "Games", "Videoblog", "People","Comedy","Entertainment","News","Howto",
            "Education","Tech","Nonprofit"};
		    EnumManager.UpdateEnum("YouTubeTags", "Tech", s);
		}

        #endregion fields & pins

        private const int ChunkSize = 25;
        private ResumableUploader youtube_uploader = null;
        private Authenticator youtube_authenticator = null;
        private GDataCredentials youtube_credentials;
        private bool isUploading = false;
        
        //called when data for any output pin is requested
        public void Evaluate(int SpreadMax)
        {
            if (FPinInDoSend.IsChanged && FPinInDoSend[0])
            {
                if (youtube_uploader == null)
                    InitYouTube();

                if(!isUploading)
                    UploadVideoToYouTube();
            }
        }

        private void InitYouTube()
        {
            try
            {
                youtube_credentials = new GDataCredentials(FUserName[0], FPassword[0]);
                youtube_authenticator = new ClientLoginAuthenticator(FApplicationName[0], ServiceNames.YouTube, youtube_credentials);
                youtube_authenticator.DeveloperKey = FDevKey[0];
                    
                youtube_uploader = new ResumableUploader(ChunkSize);
                youtube_uploader.AsyncOperationCompleted += new AsyncOperationCompletedEventHandler(OnDone);
                youtube_uploader.AsyncOperationProgress += new AsyncOperationProgressEventHandler(OnProgress);
            }
            catch (Exception ex)
            {
                FLogger.Log(LogType.Error, "Error Authenticating: " + ex.Message);
            }
        }

        private void UploadVideoToYouTube()
        { 
            Video video = new Video();
            video.Title = FVideoTitle[0];
            video.Description = FVideoDescription[0];
            video.Keywords = FVideoKeywords[0];
            video.Tags.Add(new MediaCategory(FVideoTags[0].Name, YouTubeNameTable.CategorySchema));
            video.Private = FVideoPrivate[0];

            string contentType = MediaFileSource.GetContentTypeForFileName(FVideoLink[0]);
            video.MediaSource = new MediaFileSource(FVideoLink[0], contentType);
            
            //The default portion could also be replaced by a specific channel name
            AtomLink link = new AtomLink("http://uploads.gdata.youtube.com/resumable/feeds/api/users/" + "default" + "/uploads");
            link.Rel = ResumableUploader.CreateMediaRelation;
            video.YouTubeEntry.Links.Add(link);

            UserState user_state = new UserState();
            user_state.retryCounter = 0;
            FLogger.Log(LogType.Debug, "Starting Upload of Video" + FVideoLink[0]);
            FLogger.Log(LogType.Debug, "ContentType: " + contentType);
            FLogger.Log(LogType.Debug, "Tags: " + FVideoTags[0].Name);
            isUploading = true;
            youtube_uploader.InsertAsync(youtube_authenticator, video.YouTubeEntry, user_state);
        }

        private void OnProgress(object sender, AsyncOperationProgressEventArgs e)
        {
            UserState user_state = e.UserState as UserState;

            if (user_state != null)
            {
                string status = "";
                if (user_state.retryCounter > 1)
                {
                    status = "Retrying (" + (user_state.retryCounter - 1).ToString() + "), uploading: " + e.ProgressPercentage + "% done";
                }
                else
                {
                    status = "Uploading: " + e.ProgressPercentage + "% done";
                }
                user_state.currentPosition = e.Position;
                user_state.resumeUri = e.Uri;
                user_state.httpVerb = e.HttpVerb;
                FLogger.Log(LogType.Debug, "Status: " + status);
            }
            FRequestProgress[0] = e.ProgressPercentage;
        }

        private void OnDone(object sender, AsyncOperationCompletedEventArgs e)
        {
            if (e != null)
            {
                UserState user_state = e.UserState as UserState;
                FLogger.Log(LogType.Message, "Upload completed for: " + user_state.resumeUri);
                if (user_state != null)
                {
                    if (e.Cancelled)
                    {
                        FLogger.Log(LogType.Message, "Upload was cancelled!");
                        user_state.retryCounter = 0;
                        //Function Try Again (retry que...)
                    }
                    else if (e.Error != null)
                    {
                        user_state.errorText = e.Error.Message;
                        FLogger.Log(LogType.Error, "An error occured with the upload:" + e.Error.Message);
                        //Function Try Again....
                    }
                    else
                    {
                        //parse result! finished good
                        ParseAndFinish(user_state, e.ResponseStream);
                        FFinishedUploading[0] = true;
                    }                    

                }

            }
            isUploading = false;
        }

        private void ParseAndFinish(UserState u, Stream s)
        {
            YouTubeRequestSettings ys = new YouTubeRequestSettings(FApplicationName[0], FDevKey[0]);
            YouTubeRequest ytr = new YouTubeRequest(ys);
            Video v = ytr.ParseVideo(s);

            FVideoInformationResponse[0] = "";
            FVideoInformationResponse[0] += "Title: " + v.Title;
            FVideoInformationResponse[0] += "\n"; 
            FVideoInformationResponse[0] += "Video ID: " + v.VideoId;
            FVideoInformationResponse[0] += "\n";
            FVideoInformationResponse[0] += "Keywords: " + v.Keywords;
            FVideoInformationResponse[0] += "\n";
            FVideoInformationResponse[0] += "Summary: " + v.Summary;

            FResponse[0] += v.WatchPage.AbsoluteUri;

            FLogger.Log(LogType.Message, "Video ID" + v.VideoId);
            FLogger.Log(LogType.Message, "Status: " + v.Status);
        }

    }

    class UserState
    {
        public long currentPosition;
        public string httpVerb;
        public Uri resumeUri;
        public int retryCounter;
        public string errorText;
    }
}