#ifndef RAW_INDEXEDGEOM_FXH
#define RAW_INDEXEDGEOM_FXH

#include "RawUtils.fxh"

int MaxVertexCount : MAXVERTEXCOUNT;
int MaxIndexCount : MAXINDEXCOUNT;

/*Header functions, we need : 
-Vertex Counter
-Index Counter
-Dispatch Buffer for vertices
-Dispatch Buffer for indices
-Indexed Draw Buffer*/

#define VERTEX_COUNTER_LOCATION 0
#define INDEX_COUNTER_LOCATION 4
#define VERTEX_DISPATCH_LOCATION 8
//Add 12 for index dispatch
#define INDEX_DISPATCH_LOCATION 20

#define DRAW_INDEXED_LOCATION 32
//Size of index dispatch is 20

#define POSITION_BUFFER_LOCATION 52
#define POSITION_VERTEX_STRIDE 12

//Now add N elements for each index buffer
#define INDEX_BUFFER_LOCATION (POSITION_BUFFER_LOCATION + (MaxVertexCount * POSITION_VERTEX_STRIDE))
#define TRIANGLE_INDEX_STRIDE 12

void ResetCounters()
{
	RWByteBuffer.Store(VERTEX_COUNTER_LOCATION, 0);
	RWByteBuffer.Store(INDEX_COUNTER_LOCATION, 0);
}

void SetDispatchCalls(uint groupSize)
{
	uint vertexCount = RWByteBuffer.Load(VERTEX_COUNTER_LOCATION);
	uint faceCount = RWByteBuffer.Load(INDEX_COUNTER_LOCATION);
	
	SetIndirectDispatch(VERTEX_DISPATCH_LOCATION, uint3((vertexCount+groupSize-1)/groupSize,1,1));
	SetIndirectDispatch(INDEX_DISPATCH_LOCATION, uint3((faceCount+groupSize-1)/groupSize,1,1));
}

void SetDrawCall()
{
	uint faceCount = RWByteBuffer.Load(INDEX_COUNTER_LOCATION);
	
	RWByteBuffer.Store4(DRAW_INDEXED_LOCATION, uint4(faceCount*3,1,0,0));
	RWByteBuffer.Store(DRAW_INDEXED_LOCATION+16, 0);
}

float3 ReadPosition(uint index)
{
	return ReadFloat3(POSITION_BUFFER_LOCATION + index * POSITION_VERTEX_STRIDE);
}

void WritePosition(uint index, float3 position)
{
	WriteFloat3(POSITION_BUFFER_LOCATION + index * POSITION_VERTEX_STRIDE, position);
}


uint AppendPosition(float3 position)
{
	uint index = IncrementCounter(VERTEX_COUNTER_LOCATION);
	WritePosition(index, position);
	return index;
}

 
uint AppendPositionSafe(float3 position)
{
	uint index = InterLockedAddSafe(VERTEX_COUNTER_LOCATION,1);
	WritePosition(index, position);
	return index;
}

uint3 ReadTriangle(uint index)
{
	return RWByteBuffer.Load3(INDEX_BUFFER_LOCATION + index * TRIANGLE_INDEX_STRIDE);
}

void AppendTriangle(uint3 indices)
{
	uint index = IncrementCounter(INDEX_COUNTER_LOCATION);
	RWByteBuffer.Store3(INDEX_BUFFER_LOCATION + index * TRIANGLE_INDEX_STRIDE, indices);
}



#endif