#region usings
using System;
using System.ComponentModel.Composition;
using System.Collections.Generic;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;

using VVVV.Core.Logging;

using Pathfinder.Utility;

#endregion usings

namespace VVVV.Nodes
{
	// indicates debug output
	public enum DebugMode
	{
		None,
		Angle,
		SmoothedAngle,
		d_Angle,
		dd_Angle,
		ddd_Angle,
		Kernel
	}

	#region PluginInfo
	[PluginInfo(Name = "FindCorners", Category = "Pathfinder", Help = "Basic template with one value in/out", Tags = "")]
	#endregion PluginInfo
	public class PathfinderPathInterpolationNode : IPluginEvaluate
	{
		#region Pins
		[Input("Path")]
		IDiffSpread<Vector2D> FPath;
		
		[Input("Relative Position", MinValue = 0, MaxValue = 1)]
		IDiffSpread<double> FPositionIn;

		[ConfigAttribute("Step Size", IsSingle = true, DefaultValue = 1, MinValue = 1)]
		IDiffSpread<int> FStep;
		
		[ConfigAttribute("Debug Mode")]
		IDiffSpread<DebugMode> FDebugMode;
		
		[ConfigAttribute("Kernel Size", IsSingle = true, MinValue = 3, DefaultValue = 15)]
		IDiffSpread<int> FKernelSize;
		
		[ConfigAttribute("Standard Deviation", IsSingle = true, MinValue = .5, DefaultValue = 3)]
		IDiffSpread<double> FSigma;
		
		[ConfigAttribute("Threshold", IsSingle = true, MinValue = .0001, DefaultValue = .0001)]
		IDiffSpread<double> FThreshold;

		[Output("Corners")]
		ISpread<Vector2D> FCorners;
		
		[Output("Debug")]
		ISpread<double> FDebug;
		
		[Output("Position")]
		ISpread<Vector2D> FPositionOut;

		[Import()]
		ILogger FLogger;
		#endregion Pins

		#region Fields
		
		// Stores a value between 0 and 1 for every every corner,
		// according to its relative position on the path.
		double[] FCornerPointPositions = null;
		
		// Positions for the stored corner points
		Vector2D[] FCornerPoints = null;
		
		#endregion Fields

		#region Methods
			
		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{	
			if (FPath.SliceCount == 0)
				// TODO: clear output
				return; 
				
			// only recalculate the path if it has changed
			if (	FPath.IsChanged || 
					FKernelSize.IsChanged || 
					FSigma.IsChanged || 
					FThreshold.IsChanged ||
					FStep.IsChanged ||
					FDebugMode.IsChanged)
				Recalculate();

			// only redo interpolation if relative posiion has changed 
			if (FPositionIn.IsChanged)
				Interpolate();
		}
		
		// Recalculates the input path and updates the "Corners" output.
		// Calls Interpolate() afterwards.
		private void Recalculate()
		{
			if (FPath.SliceCount == 0)
			{
				FPositionOut.SliceCount = 0;
				FCorners.SliceCount = 0;
			}
		
			// get path form input
			Vector2D[] pathNonUnit = new Vector2D[FPath.SliceCount];
			for (int i = 0; i < FPath.SliceCount; i += FStep[0])
				pathNonUnit[i] = FPath[i];
	
			// calculate path length
			Vector2D[] pathDifference = Utility.Differentiate(pathNonUnit, null);			
			
			// get angles
			double[] anglesRaw = Utility.GetAngles(pathDifference);
			double[] angles = new double[anglesRaw.Length];
			angles[0] = anglesRaw[0];
			
			// ensure continuity (2-pi-wrap-around)
			double change = 0;
			for (int i = 0; i < anglesRaw.Length - 1; ++i)
			{
				double diffAngle = anglesRaw[i + 1] - anglesRaw[i];
				//FLogger.Log(LogType.Debug, diffAngle.ToString());
				if (diffAngle > Math.PI)
					change += -Math.PI * 2;					
					
				if (diffAngle < -Math.PI)
					change += Math.PI * 2;
					
				angles[i + 1] = anglesRaw[i + 1] + change;
			}

			// smooth angles by convolving them with a gaussian kernel
			double[] kernel = Utility.GenerateGaussianKernel(FSigma[0], FKernelSize[0]);
			double[] anglesSmoothed = Utility.Filter(angles, kernel);															
			
			// get derrivates
			double[] d_angles = Utility.Differentiate(anglesSmoothed, null);
			for (int i = 0; i < d_angles.Length; ++i)
				d_angles[i] = d_angles[i] * d_angles[i];
			
			double[] dd_angles = Utility.Differentiate(d_angles, null);
			double[] ddd_angles = Utility.Differentiate(dd_angles, null);
			
			// find zero crossings for 2nd derrivate
			List<int> zeroIDs = new List<int>();
			for (int i = 0; i < dd_angles.Length - 1; ++i)
				if (	dd_angles[i] * dd_angles[i+1] < 0 && 
						ddd_angles[i] < 0 &&
						d_angles[i] > FThreshold[0] / 1000)
					zeroIDs.Add(i); 
			
			// set corners array
			FCornerPoints = new Vector2D[zeroIDs.Count + 2];
			FCornerPoints[0] = FPath[0];		
			for (int i = 0; i < zeroIDs.Count; ++i)
				FCornerPoints[i+1] = FPath[zeroIDs[i] + 2];
			FCornerPoints[FCornerPoints.Length-1] = FPath[FPath.SliceCount-1];
			
			// set output
			FCorners.SliceCount = FCornerPoints.Length;
			for (int i = 0; i < FCornerPoints.Length; ++i)
				FCorners[i] = FCornerPoints[i];
			
			PrepareInterpolation();			
		
			#region debug out
			// debug out
			switch (FDebugMode[0])
			{
				case DebugMode.None:
					FDebug.SliceCount = 0;
					break;
					
				case DebugMode.Angle:
					SetDebugOut(angles);
					break;
					
				case DebugMode.SmoothedAngle:
					SetDebugOut(anglesSmoothed);				
					break;
					
				case DebugMode.d_Angle:
					SetDebugOut(d_angles);
					break;
						
				case DebugMode.dd_Angle:
					SetDebugOut(dd_angles);
					break;			
					
				case DebugMode.ddd_Angle:
					SetDebugOut(ddd_angles);
					break;	
				
				case DebugMode.Kernel:
					SetDebugOut(kernel);
					break;	
			}
			#endregion				
           						
			Interpolate();
		}

		private void PrepareInterpolation()
		{
					// find steplength between corners and overall length
			double[] stepLengths = Utility.GetLengths(Utility.Differentiate(FCornerPoints, null));
			double pathLength = 0;
			foreach (double step in stepLengths)
				pathLength += step;
		
			FCornerPointPositions = new double[stepLengths.Length + 1];
			FCornerPointPositions[0] = 0;
			for (int i = 1; i < stepLengths.Length; ++i)
				FCornerPointPositions[i] = FCornerPointPositions[i-1] + stepLengths[i-1] / pathLength;		
			FCornerPointPositions[stepLengths.Length] = 1;
		}
		
		// Interpolates between the corner positions according to the relative
		// position and changes the "Positions" output afterwards.		
		private void Interpolate()
		{
			// find corresponding corner index
			int i;
			for (i = 0; i < FCornerPointPositions.Length - 1; ++i)
				if (FCornerPointPositions[i] > FPositionIn[0])	
					break;

			// get interval offset
			double full = FCornerPointPositions[i] - FCornerPointPositions[i - 1];
			double frac = FPositionIn[i] - FCornerPointPositions[i - 1];

			// set output
			Vector2D vec = FCornerPoints[i] - FCornerPoints[i - 1];						
			FPositionOut.SliceCount = 1;
			FPositionOut[0] = FCornerPoints[i - 1] + vec * (frac / full);
			
		}
	
		private void SetDebugOut(double[] values)
		{
			FDebug.SliceCount = values.Length;
			
			// get min, max values
			double min = double.MaxValue;
			double max = double.MinValue;
			for (int i=0; i<values.Length; ++i)
			{
				if (min > values[i]) min = values[i];
				if (max < values[i]) max = values[i];
			}
			
			// normalize
			double range = max - min;
			for (int i=0; i<values.Length; ++i)
				FDebug[i] = (values[i] - min) / range;
		}
	
		#endregion Methods
	}
}
