using System;

using VVVV.Utils.VMath;

namespace Pathfinder.Utility
{
	public enum SignalExtension
	{
		Zero,
		One,
		Mirror,
		Copy
	}
	
	public class Utility
	{		
		// Calculates the discrete derrivate of a given function of type Vector2D
		// For step == null unit steps are assumed
		public static Vector2D[] Differentiate(Vector2D[] func, double[] step)
		{
			Vector2D[] result = new Vector2D[func.Length - 1];
			
			if (step == null)
				for (int i = 0; i < result.Length; ++i)
					result[i] = func[i+1] - func[i];
			else
				for (int i = 0; i < result.Length; ++i)
					result[i] = (func[i+1] - func[i]) / step[i];
			
			return result;
		}
		
		// Calculates the discrete derrivate of a given function of type double
		// For step == null unit steps are assumed
		public static double[] Differentiate(double[] func, double[] step)
		{			
			double[] result = new double[func.Length - 1];
			
			if (step == null)
				for (int i = 0; i < result.Length; ++i)
					result[i] = func[i+1] - func[i];
			else
				for (int i = 0; i < result.Length; ++i)
					result[i] = (func[i+1] - func[i]) / step[i];
			
			return result;
		}
	
		// Calculates the angles from the x-axis for the given vectors
		public static double[] GetAngles(Vector2D[] vectors)
		{
			double[] result = new double[vectors.Length];
			
			for (int i = 0; i < result.Length; ++i)
				result[i] = Math.Atan2(vectors[i].y, vectors[i].x);				
				
			return result;
		}
		
		// Calculates the lengths for the given vectors
		public static double[] GetLengths(Vector2D[] vectors)
		{
			double[] result = new double[vectors.Length];
			
			for (int i = 0; i < result.Length; ++i)
				result[i] = Math.Sqrt(vectors[i].x * vectors[i].x + vectors[i].y * vectors[i].y);
				
			return result;
		}
		
		// Calculates a gaussian filter kernel for a given standart deviation and 
		// kernel size.
		public static double[] GenerateGaussianKernel(double sigma, int kernelSize)
		{
			// ensure meaningfull kernel size and sigma
			if (kernelSize < 3) kernelSize = 3;				
			if (sigma < .01) sigma = .01;
				
			// prepare calculation
			double[] result = new double[kernelSize];
			double v1 = 2 * sigma * sigma;
			double v2 = v1 * Math.PI;
			double kernelOffset = Math.Floor((double)kernelSize / 2);
			double kernelSum = 0;
			
			// create kernel
			for (int i = 0; i < kernelSize; ++i)
			{				
				double arg = (double)i - kernelOffset;
				result[i] = Math.Exp(-arg * arg / v1) / v2;
				kernelSum += result[i]; 
			}
			
			// normalize kernel
			for (int i = 0; i < kernelSize; ++i)
				result[i] /= kernelSum;						
			
			return result;
		}
		
		public static double[] Filter(double[] signal, double[] kernel)
		{			
			double[] result = new double[signal.Length];
			int kernelOffset = kernel.Length / 2;
			int maxIndex = result.Length - 1;
			
			for (int i = 0; i < result.Length; ++i)
			{
				result[i] = 0;
				for (int j = 0; j < kernel.Length; ++j)
				{
					// clamp index 
					int index = Math.Min(Math.Max(i + j - kernelOffset, 0), maxIndex);				
					result[i] += kernel[j] * signal[index];
				}
			}
			
			return result;
		}		
	}
}