///////////////////////////////////////////////////////////////////////////////////
// FreeFrameSample.cpp
//
// FreeFrame Open Video Plugin 
// C Version
//
// Implementation of the Free Frame sample plugin
//
// www.freeframe.org
// marcus@freeframe.org

/*

Copyright (c) 2002, Marcus Clements www.freeframe.org
All rights reserved.

FreeFrame 1.0 upgrade by Russell Blakeborough
email: boblists@brightonart.org

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in
     the documentation and/or other materials provided with the
     distribution.
   * Neither the name of FreeFrame nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//includes

#include "Contour.h"
#include <string.h>
#include <stdio.h>


// Plugin Globals
PlugInfoStruct GPlugInfo;
PlugExtendedInfoStruct GPlugExtInfo;
ParamConstsStruct GParamConstants[NUM_PARAMS];
OutputConstsStruct GOutputConstants[NUM_OUTPUTS];

LPVOID instantiate(VideoInfoStruct* pVideoInfo)
{
	// Create local pointer to plugObject
	plugClass *pPlugObj;
	// create new instance of plugClass
	pPlugObj = new plugClass;

	// make a copy of the VideoInfoStruct
	pPlugObj->FVideoInfo.frameWidth = pVideoInfo->frameWidth;
	pPlugObj->FVideoInfo.frameHeight = pVideoInfo->frameHeight;
	pPlugObj->FVideoInfo.bitDepth = pVideoInfo->bitDepth;

	// this shouldn't happen if the host is checking the capabilities properly
	pPlugObj->FVideoMode = pPlugObj->FVideoInfo.bitDepth;
	if (pPlugObj->FVideoMode >2 || pPlugObj->FVideoMode < 0) {
	  return (LPVOID) FF_FAIL;
	}

	pPlugObj->init();
	
	// Russell - return pointer to the plugin instance object we have created

	// return pointer to object cast as LPVOID 
	return (LPVOID) pPlugObj;
}

DWORD deInstantiate(LPVOID instanceID)
{
	// declare pPlugObj - pointer to this instance
	plugClass *pPlugObj;

	// typecast LPVOID into pointer to a plugClass
	pPlugObj = (plugClass*) instanceID;

	delete pPlugObj; // todo: ? success / fail?
	
	return FF_SUCCESS;
}

DWORD initialise()
{
    // populate the parameters constants structs
    GParamConstants[0].Type = 10;
    GParamConstants[1].Type = 10;
    GParamConstants[2].Type = 0;
    
	GParamConstants[0].Default = 10.0f;
	GParamConstants[1].Default = 0.0f;
	GParamConstants[2].Default = false;

	char tempName0[17] = "buffer size";
	char tempName1[17] = "clamp";
	char tempName2[17] = "binary";
	
	memcpy(GParamConstants[0].Name, tempName0, 16);
	memcpy(GParamConstants[1].Name, tempName1, 16);
	memcpy(GParamConstants[2].Name, tempName2, 16);
	
	return FF_SUCCESS;
}

DWORD deInitialise()
{
	return FF_SUCCESS;
}

DWORD getNumParameters()
{
	return NUM_PARAMS;  
}

DWORD getNumOutputs()
{
	return NUM_OUTPUTS;  
}

char* getParameterName(DWORD index)
{
	return GParamConstants[index].Name;
}

char* getOutputName(DWORD index)
{
	return GOutputConstants[index].Name;
}

float getParameterDefault(DWORD index)
{
	return GParamConstants[index].Default;
}

unsigned int getParameterType(DWORD index)
{
	return GParamConstants[index].Type;
}

unsigned int getOutputType(DWORD index)
{
	return GOutputConstants[index].Type;
}


plugClass::plugClass() : _isInit(false), _ringBuf(NULL)
{   
    InitializeCriticalSection(&CriticalSection); 
}

void plugClass::init()
{
    FImageSize.width = FVideoInfo.frameWidth;
    FImageSize.height = FVideoInfo.frameHeight;

    _frameImageFloat = cvCreateImage(FImageSize, IPL_DEPTH_32F, 1);
    _lastImageFloat = cvCreateImageHeader(FImageSize, IPL_DEPTH_32F, 1);
    
    _currentImage = cvCreateImageHeader(FImageSize, IPL_DEPTH_8U, 3);
    
    _grayImage = cvCreateImage(FImageSize, IPL_DEPTH_8U, 1);
    _floatImage = cvCreateImage(FImageSize, IPL_DEPTH_32F, 1);
    
//    _ringBuf = new RingBuffer( 10, _floatImage->imageSize );
}

plugClass::~plugClass()
{
    delete _ringBuf;
    
    cvReleaseImage( &_frameImageFloat );
    cvReleaseImage( &_grayImage );
    cvReleaseImage( &_floatImage );
                       
    DeleteCriticalSection(&CriticalSection);
}


char* plugClass::getParameterDisplay(DWORD index)
{
	// fill the array with spaces first
	for (int n=0; n<16; n++) {
		FParams[index].DisplayValue[n] = ' ';
	}
	sprintf(FParams[index].DisplayValue, "%f",FParams[index].Value);
	return FParams[index].DisplayValue;
}


DWORD plugClass::setParameter(SetParameterStruct* pParam)
{
      if (!_isInit)
      {
         delete _ringBuf;
         _ringBuf = new RingBuffer( (unsigned char)FParams[0].Value , _frameImageFloat->imageSize);
        
      }
      
    //only set value if really changed..should better be checked in host!!
    if (pParam->value == FParams[pParam->index].Value) 
        return FF_SUCCESS;
    else
    {
        EnterCriticalSection(&CriticalSection);
        
        if (pParam->index == 0)
        {
            if (pParam->value >=2)
               FParams[pParam->index].Value = pParam->value;
            else
               FParams[pParam->index].Value = 2; 
            
        	if ( _isInit )
            {
                 delete _ringBuf;
                 _ringBuf = new RingBuffer( (unsigned char)FParams[0].Value , _frameImageFloat->imageSize);
        
                 cvSet( _frameImageFloat, cvScalar(0) );
            }
        } else
          FParams[pParam->index].Value = pParam->value;
            
        LeaveCriticalSection(&CriticalSection);
    	
    	
    	return FF_SUCCESS;
   	} 
}

float plugClass::getParameter(DWORD index)
{    
	return FParams[index].Value;
}

DWORD plugClass::getOutputSliceCount(DWORD index)
{
	return FOutputs[index].SliceCount;
}

#define PI 3.14159265

float* plugClass::getOutput(DWORD index)
{
    EnterCriticalSection(&CriticalSection);
           
    LeaveCriticalSection(&CriticalSection); 
    
	return FOutputs[index].Spread;
}

DWORD plugClass::processFrame(LPVOID pFrame)
{
	switch (FVideoInfo.bitDepth) {
		case 1:
			return processFrame24Bit(pFrame);
		case 2:
			return processFrame32Bit(pFrame);
		default:
			return FF_FAIL;
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////

DWORD plugClass::processFrame24Bit(LPVOID pFrame)
{
    EnterCriticalSection(&CriticalSection);
    _isInit = true;  
    
    if (_ringBuf)
    {
       _currentImage->imageData = static_cast<char*>( pFrame );
    
       cvCvtColor( _currentImage, _grayImage, CV_BGR2GRAY );
       cvConvert( _grayImage, _floatImage );
    
       IplImage *origImage = cvCloneImage( _floatImage );
    
       cvScale( _floatImage, _floatImage, 1.0/FParams[0].Value);
    
       _ringBuf->insert( _floatImage->imageData );
    
       _lastImageFloat->imageData = static_cast<char*>(_ringBuf->getLast());
       cvSub( _frameImageFloat, _lastImageFloat, _frameImageFloat );
       cvAdd( _frameImageFloat, _floatImage, _frameImageFloat );
    
       cvSub( _frameImageFloat, origImage, origImage );
    
       cvConvert( origImage, _grayImage );
       
       if ( FParams[2].Value )
           cvThreshold( _grayImage, _grayImage, FParams[1].Value * 255, 255, CV_THRESH_BINARY );
       else
           cvThreshold( _grayImage, _grayImage, FParams[1].Value * 255, 255, CV_THRESH_TOZERO );
        
       cvCvtColor( _grayImage, _currentImage, CV_GRAY2BGR );
  
       cvReleaseImage( &origImage );
    }
    
    LeaveCriticalSection(&CriticalSection);
        
	return FF_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////

DWORD plugClass::processFrame32Bit(LPVOID pFrame)
{
	return FF_FAIL;
}

DWORD plugClass::processFrameCopy(ProcessFrameCopyStruct* pFrameData)
{
    return FF_FAIL;
}

DWORD plugClass::processFrameCopy24Bit(ProcessFrameCopyStruct* pFrameData)
{
	return FF_FAIL;
}

DWORD plugClass::processFrameCopy32Bit(ProcessFrameCopyStruct* pFrameData)
{
	return FF_FAIL;
}

DWORD getPluginCaps(DWORD index)
{
	switch (index) {

	case FF_CAP_16BITVIDEO:
		return FF_FALSE;

	case FF_CAP_24BITVIDEO:
		return FF_TRUE;

	case FF_CAP_32BITVIDEO:
		return FF_TRUE;

	case FF_CAP_PROCESSFRAMECOPY:
		return FF_FALSE;

	case FF_CAP_MINIMUMINPUTFRAMES:
		return NUM_INPUTS;

	case FF_CAP_MAXIMUMINPUTFRAMES:
		return NUM_INPUTS;

	case FF_CAP_COPYORINPLACE:
		return FF_FALSE;

	default:
		return FF_FALSE;
	}
}


PlugInfoStruct* getInfo() 
{
	GPlugInfo.APIMajorVersion = 2;		// number before decimal point in version nums
	GPlugInfo.APIMinorVersion = 000;		// this is the number after the decimal point
										// so version 0.511 has major num 0, minor num 501
	char ID[5] = "hloA";		 // this *must* be unique to your plugin 
								 // see www.freeframe.org for a list of ID's already taken
	char name[17] = "frameBuf";
	
	memcpy(GPlugInfo.uniqueID, ID, 4);
	memcpy(GPlugInfo.pluginName, name, 16);
	GPlugInfo.pluginType = FF_EFFECT;

	return &GPlugInfo;
}

LPVOID getExtendedInfo()
{

	GPlugExtInfo.PluginMajorVersion = 1;
	GPlugExtInfo.PluginMinorVersion = 10;

	// I'm just passing null for description etc for now
	// todo: send through description and about
	GPlugExtInfo.Description = NULL;
	GPlugExtInfo.About = NULL;

	// FF extended data block is not in use by the API yet
	// we will define this later if we want to
	GPlugExtInfo.FreeFrameExtendedDataSize = 0;
	GPlugExtInfo.FreeFrameExtendedDataBlock = NULL;

	return (LPVOID) &GPlugExtInfo;
}
