//@author: dottore
//@description: Gooch-style "diffuse" with a glossy vertex-applied shading. The
//		highlight is done in VERTEX shading -- not as a texture.
//              Glossiness is controlled not only by the usual power function, but also
//              by applying a set of gloss controls that cause a sharp falloff across
//	        a specified range.
//              The falloff will occur in the highlight range [glossBot-glossTop] and the
//              amount of falloff is specified by "glossDrop." Setting "glossDrop"
//	        to 1.0 nullifies the effect.
//@tags: goochy directional glossy vertex shading
//@credits: nVidia

/*********************************************************************NVMH3****

Copyright NVIDIA Corporation 2002-2004
TO THE MAXIMUM EXTENT PERMITTED BY APPLICABLE LAW, THIS SOFTWARE IS PROVIDED
*AS IS* AND NVIDIA AND ITS SUPPLIERS DISCLAIM ALL WARRANTIES, EITHER EXPRESS
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL NVIDIA OR ITS SUPPLIERS
BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT, OR CONSEQUENTIAL DAMAGES
WHATSOEVER (INCLUDING, WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR ANY OTHER PECUNIARY LOSS)
ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF NVIDIA HAS
BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.


******************************************************************************/

/************* TWEAKABLES **************/

float4x4 WorldIT : WorldInverseTranspose < string UIWidget="None"; >;
float4x4 WorldViewProj : WorldViewProjection < string UIWidget="None"; >;
float4x4 World : World < string UIWidget="None"; >;
float4x4 ViewI : ViewInverse < string UIWidget="None"; >;

float3 LightDir : Direction
<
	string Object = "DirectionalLight";
	string Space = "World";
> = {100.0f, 100.0f, 100.0f};

float4 LiteColor : COLOR <
    string UIName =  "Bright Surface Color";
    string UIWidget = "Color";
> = {0.8f, 0.5f, 0.1f, 1.0f};

float4 DarkColor : COLOR <
    string UIName =  "Dark Surface Color";
    string UIWidget = "Color";
> = {0.0f, 0.0f, 0.0f, 1.0f};

float4 WarmColor : COLOR<
    string UIName =  "Gooch warm tone";
    string UIWidget = "Color";
> = {0.5f, 0.4f, 0.05f, 1.0f};

float4 CoolColor : COLOR<
    string UIName =  "Gooch cool tone";
    string UIWidget = "Color";
> = {0.05f, 0.05f, 0.6f, 1.0f};

float4 SpecColor : COLOR <
    string UIName =  "Hilight color";
    string UIWidget = "Color";
> = {0.7f, 0.7f, 1.0f, 1.0f};

float SpecExpon : SpecularPower <
    string UIWidget = "slider";
    float UIMin = 1.0;
    float UIMax = 128.0;
    float UIStep = 1.0;
    string UIName =  "specular power";
> = 40.0;

float GlossTop <
    string UIWidget = "slider";
    float UIMin = 0.2;
    float UIMax = 1.0;
    float UIStep = 0.05;
    string UIName =  "Maximum for Gloss Dropoff";
> = 0.7;

float GlossBot
<
    string UIWidget = "slider";
    float UIMin = 0.05;
    float UIMax = 0.95;
    float UIStep = 0.05;
    string UIName =  "Minimum for Gloss Dropoff";
> = 0.5;

float GlossDrop
<
    string UIWidget = "slider";
    float UIMin = 0.0;
    float UIMax = 1.0;
    float UIStep = 0.05;
    string UIName =  "Strength of Glossy Dropoff";
> = 0.2;

/************* DATA STRUCTS **************/

/* data from application vertex buffer */
struct appdata {
    float3 Position	: POSITION;
    float4 UV		: TEXCOORD0;
    float4 Normal	: NORMAL;
};

/* data passed from vertex shader to pixel shader */
struct vertexOutput {
    float4 HPosition	: POSITION;
    float4 TexCoord0	: TEXCOORD0;
    float4 diffCol	: COLOR0;
    float4 specCol	: COLOR1;
};

/*********** vertex shader ******/

vertexOutput gooch2VS(appdata IN) {
    vertexOutput OUT;
    float3 Nn = normalize(mul(IN.Normal, WorldIT).xyz);
    float4 Po = float4(IN.Position.xyz,1);
    //compute worldspace position
    float3 Pw = mul(Po, World).xyz;
    float3 Ln = normalize(-LightDir);
    float mixer = 0.5 * (dot(Ln,Nn) + 1.0);
    float4 surfColor = lerp(DarkColor,LiteColor,mixer);
    float4 toneColor = lerp(CoolColor,WarmColor,mixer);
    float4 mixColor = surfColor + toneColor;
    mixColor.w = 1.0;
    OUT.diffCol = mixColor;
    OUT.TexCoord0 = IN.UV;
    float3 Vn = normalize(ViewI[3].xyz - Pw);
    float3 Hn = normalize(Vn + Ln);
    float hdn = pow(max(0,dot(Hn,Nn)),SpecExpon);
    hdn = hdn * (GlossDrop+smoothstep(GlossBot,GlossTop,hdn)*(1.0-GlossDrop));
    OUT.specCol = hdn * SpecColor;
    OUT.HPosition = mul(Po, WorldViewProj);
    return OUT;
}

/********* pixel shader ********/

float4 gooch2PS(vertexOutput IN) : COLOR {
    return (IN.diffCol + IN.specCol);
}

/*************/

technique PixelShaderVersion <
	string Script = "Pass=p0;";
> {
    pass p0 <
	string Script = "Draw=geometry;";
> {		
		VertexShader = compile vs_1_1 gooch2VS();
		ZEnable = true;
		ZWriteEnable = true;
		CullMode = None;
		PixelShader = compile ps_1_1 gooch2PS();
    }
}

technique NoPixelShaderVersion <
	string Script = "Pass=p0;";
> {
    pass p0 <
	string Script = "Draw=geometry;";
> {		
		VertexShader = compile vs_1_1 gooch2VS();
		ZEnable = true;
		ZWriteEnable = true;
		CullMode = None;
		SpecularEnable = true;
		ColorArg1[ 0 ] = Diffuse;
		ColorOp[ 0 ]   = SelectArg1;
		ColorArg2[ 0 ] = Specular;
		AlphaArg1[ 0 ] = Diffuse;
		AlphaOp[ 0 ]   = SelectArg1;
    }
}

/***************************** eof ***/
