#region usings
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;
using VVVV.Utils.Animation;

using VVVV.Core.Logging;
#endregion usings

namespace VVVV.Nodes
{
	public enum AttractionForce
	{
		Eades,
		FruchtermanReingold,
		Spring
	}
	
	public enum RepulsionForce
	{
		Eades,
		FruchtermanReingold,
		Attractor
	}
	
	#region PluginInfo
	[PluginInfo(Name = "ForceDirected", 
				Category = "Graph", 
				Help = "Creates a force based layout of a data graph.", 
				Tags = "layout",
				Author = "tonfilm")]
	#endregion PluginInfo
	public class ForceDirected : IPluginEvaluate
	{
		#region fields & pins
		[Input("Root ID", IsSingle = true)]
		IDiffSpread<string> FRootID;
		
		[Input("From ID", DefaultString = "A")]
		IDiffSpread<string> FFromID;
		
		[Input("To ID", DefaultString = "B")]
		IDiffSpread<string> FToID;
		
		[Input("Edge Name", DefaultString = "A to B")]
		IDiffSpread<string> FEdgeName;
		
		[Input("Speed", DefaultValue = 0.9, IsSingle = true)]
		IDiffSpread<double> FSpeed;
		
		[Input("Is 2d", IsSingle = true)]
		IDiffSpread<bool> FIs2d;
		
		[Input("Constants Eades ", DefaultValues = new double[3]{2, 1, 1})]
		IDiffSpread<Vector3D> FConstantsEades;
		
		[Input("Constants Fruchterman-Reingold ", DefaultValues = new double[2]{1, 0.7})]
		IDiffSpread<Vector2D> FConstantsFruchtermanReingold;
		
		[Input("Constants Spring-Attractor ", DefaultValues = new double[4]{1, 3, 1, 1})]
		IDiffSpread<Vector4D> FConstantsSpringAttractor;
		
		[Input("Attraction Force", IsSingle = true)]
		IDiffSpread<AttractionForce> FAttractionForce;
		
		[Input("Repulsion Force", IsSingle = true)]
		IDiffSpread<RepulsionForce> FRepulsionForce;
		
		[Input("Reset", IsBang = true, IsSingle = true)]
		ISpread<bool> FReset;

		[Output("Positions ")]
		ISpread<Vector3D> FPosOut;
		
		[Output("IDs ")]
		ISpread<string> FIDOut;
		
		[Output("From ")]
		ISpread<Vector3D> FFromOut;
		
		[Output("To ")]
		ISpread<Vector3D> FToOut;
		
		[Output("From ID")]
		ISpread<string> FFromIDOut;
		
		[Output("To ID")]
		ISpread<string> FToIDOut;
		
		[Output("Edge Name")]
		ISpread<string> FEdgeNameOut;

		[Import()]
		ILogger Flogger;
		
		[Import()]
		IHDEHost FHDEHost;
		
		bool FFirstFrame = true;
		
		#endregion fields & pins
		
		private FDGraph FGraph;
		
		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			//create graph
			if(FReset[0] || FFirstFrame || FFromID.IsChanged || FToID.IsChanged || FEdgeName.IsChanged)
			{
				FGraph = new FDGraph(FHDEHost);
				SetConsts();
				
				Forces.Attraction = FAttractionForce[0];
				Forces.Repulsion = FRepulsionForce[0];
				
				BuildGraph();
				
				Flogger.Log(LogType.Debug, "Reset");
			}
			
			//return if there is no graph
			if(FGraph == null)
			{
				FPosOut.SliceCount = 0;
				FFromOut.SliceCount = 0;
				FToOut.SliceCount = 0;
				return;
			}
				
			//update graph
			if(FSpeed.IsChanged ||
			   FRootID.IsChanged ||
			   FIs2d.IsChanged ||
			   FConstantsEades.IsChanged ||
			   FConstantsFruchtermanReingold.IsChanged ||
			   FConstantsSpringAttractor.IsChanged)
			{
				SetConsts();
			}
			
			if(FAttractionForce.IsChanged || FRepulsionForce.IsChanged)
			{
				Forces.Attraction = FAttractionForce[0];
				Forces.Repulsion = FRepulsionForce[0];
			}
			
			if(FFromID.IsChanged || FToID.IsChanged || FEdgeName.IsChanged)
			{
				BuildGraph();
			}
			
			FGraph.Update();
			
			//copy data to outputs
			FPosOut.SliceCount = FGraph.Nodes.Count;
			FIDOut.SliceCount = FGraph.Nodes.Count;
			FFromOut.SliceCount = FGraph.Edges.Count;
			FToOut.SliceCount = FGraph.Edges.Count;
			FFromIDOut.SliceCount = FGraph.Edges.Count;
			FToIDOut.SliceCount = FGraph.Edges.Count;
			FEdgeNameOut.SliceCount = FGraph.Edges.Count;
			
			for(int i=0; i<FGraph.Nodes.Count; i++)
			{
				FPosOut[i] = FGraph.Nodes[i].Position;
				FIDOut[i] = FGraph.Nodes[i].ID;
			}
			
			for(int i=0; i<FGraph.Edges.Count; i++)
			{
				var e = FGraph.Edges[i];
				FFromOut[i] = e.From.Position;
				FToOut[i] = e.To.Position;
				FFromIDOut[i] = e.From.ID;
				FToIDOut[i] = e.To.ID;
				FEdgeNameOut[i] = e.Name;
			}
			
					
			FFirstFrame = false;
		}
		
		protected void SetConsts()
		{
			FGraph.Speed = FSpeed[0];
			FGraph.RootID = FRootID[0];
			FGraph.Is2d = FIs2d[0];
			Forces.ConstantsEades = FConstantsEades[0];
			Forces.ConstantsFR = FConstantsFruchtermanReingold[0];
			Forces.ConstantsSA = FConstantsSpringAttractor[0];
		}
		
		protected void BuildGraph()
		{
			var max = Math.Max(FFromID.SliceCount, FToID.SliceCount);
			
			//add nodes
			for(int i=0; i<max; i++)
			{
				FGraph.AddEdge(FFromID[i], FToID[i], FEdgeName[i]);
			}
			
			//remove nodes
			for(int i = FGraph.Nodes.Count - 1; i >= 0; i--)
			{
				var id = FGraph.Nodes[i].ID;
				
				if(IDExists(id, FToID)) continue;
				if(IDExists(id, FFromID)) continue;
				
				FGraph.RemoveNode(FGraph.Nodes[i]);
			}
		}
		
		protected bool IDExists(string id, ISpread<string> spread)
		{
			for(int i=0; i<spread.SliceCount; i++)
			{
				if(spread[i] == id) return true;		
			}
			return false;
		}
	}
	
	//the force directed graph
	public class FDGraph
	{
		private List<FDNode> FNodes;
		private List<FDEdge> FEdges;
		public double Speed;
		public string RootID;
		private IHDEHost FHost;
		private bool FIs2d;
		
		public FDGraph(IHDEHost h)
		{
			FHost = h;
			FNodes = new List<FDNode>();
			FEdges = new List<FDEdge>();
		}
		
		//add an edge by two ids
		public void AddEdge(string f, string t, string name)
		{
			if(f == t) return;
			if(EdgeExists(f, t)) return;
		
			var fn = GetNode(f);
			var tn = GetNode(t);
			
			if(fn == null)
			{
				fn = new FDNode(f, GetTime(), IDToPos(f));
				FNodes.Add(fn);
			}
			
			if(tn == null)
			{
				tn = new FDNode(t, GetTime(), fn.Position + IDToPos(t));
				FNodes.Add(tn);
			}
			
			FEdges.Add(new FDEdge(fn, tn, name));
			
		}
		
		//get node by id
		public FDNode GetNode(string id)
		{
			var nodeCount = FNodes.Count;
			for(int i=0; i<nodeCount; i++)
			{
				if(FNodes[i].ID == id) return FNodes[i];
			}
			return null;
		}
		
		public bool EdgeExists(string f, string t)
		{
			var edgeCount = FEdges.Count;
			for(int i=0; i<edgeCount; i++)
			{
				var e = FEdges[i];
				if((e.From.ID == f && e.To.ID == t) ||
				   (e.From.ID == t && e.To.ID == f)) return true;
				   
			}
			return false;
		}
		
		public void RemoveNode(FDNode n)
		{
			foreach(var e in n.Parent)
			{
				e.From.Children.Remove(e);
				FEdges.Remove(e);
			}
			foreach(var e in n.Children)
			{
				e.To.Parent.Remove(e);
				FEdges.Remove(e);
			}
			FNodes.Remove(n);
		}
		
		public List<FDNode> Nodes
		{
			get{return FNodes;}
		}
		
		public List<FDEdge> Edges
		{
			get{return FEdges;}
		}
		
		//can be 2d
		public bool Is2d
		{
			get
			{
				return FIs2d;
			}
			set
			{
				FIs2d = value;
				
				var nodeCount = FNodes.Count;
				if(FIs2d)
				{
					for(int i=0; i<nodeCount; i++)
					{
						FNodes[i].Position.z = 0;
						FNodes[i].Velocity.z = 0;
					}
				}
				else
				{
					for(int i=0; i<nodeCount; i++)
					{
						var n = FNodes[i];
						if(n.Position.z == 0)
							n.Position.z = IDToPos(n.ID).z * 0.01;
					}
				}
			}
		}
		
		public void Update()
		{
			var edgeCount = FEdges.Count;
			var nodeCount = FNodes.Count;
			
			//attraction along edges
			for(int i=0; i<edgeCount; i++)
			{
				FEdges[i].Update();
			}
			
			var nodeCountRep = 1.0/nodeCount;
			
			//repulsion of the nodes
			for(int i=0; i<nodeCount; i++)
			{
				var n = FNodes[i];
				n.Velocity += CalcRepulsion(n);
				n.Velocity *= Speed;
			}
			
			//update position
			for(int i=0; i<nodeCount; i++)
			{
				FNodes[i].Update(GetTime());
			}
		}
		
		private Vector3D CalcRepulsion(FDNode n)
		{
			var rep = new Vector3D(0);
		
			if(n.Parent.Count == 0) return rep;
			
			var oCount = FNodes.Count;
			for(int i=0; i<oCount; i++)
			{
				var o = FNodes[i];
				
				if (o != n)
				{
					var diff = n.Position - o.Position;
					rep += Forces.Repulse(diff);
				}
			}
			
			return rep/oCount;
		}
		
		//helpers
		protected double GetTime()
		{
			return FHost.GetCurrentTime();
		}
		
		protected Vector3D IDToPos(string id)
		{
			var hash = id.GetHashCode();
			var pos = new Vector3D(0);
			
			if(id == RootID) return pos;
			
			unchecked
			{
				pos.x = ((hash & 0x00FF0000) >> 16)/255.0;
				pos.y = ((hash & 0x0000FF00) >> 8)/255.0;
				pos.z = ((hash & 0x000000FF))/255.0;
			}
			
			pos = pos * 4 - 2;
			
			if(FIs2d) pos.z = 0;
			
			return pos;
		}
	}
	
	//a node, inherits from particle
	public class FDNode : Particle
	{
		public string ID;
		public List<FDEdge> Parent;
		public List<FDEdge> Children;
		
		public FDNode(string id, double time, Vector3D pos)
			: base(time, -1, pos, new Vector3D(0))
		{
			Children = new List<FDEdge>();
			Parent = new List<FDEdge>();
			ID = id;
		}
	}
	
	//a graph edge
	public class FDEdge
	{
		public FDNode From;
		public FDNode To;
		public string Name;
		
		public FDEdge(FDNode f, FDNode t, string name)
		{
			From = f;
			To = t;
			Name = name;
			From.Children.Add(this);
			To.Parent.Add(this);
		}
		
		public void Update()
		{
			var att = Forces.Attract(From.Position - To.Position);
				
			From.Velocity -= From.Parent.Count == 0 ? new Vector3D(0) : att;
			To.Velocity += att;
		}
	}
	
	//the force functions
	public static class Forces
	{
		//constants
		public static AttractionForce Attraction;
		public static RepulsionForce Repulsion;
		
		public static Vector3D ConstantsEades;
		public static Vector2D ConstantsFR;
		public static Vector4D ConstantsSA;
		
		//public force functions
		public static Vector3D Attract(Vector3D diff)
		{
			switch(Attraction)
			{
				case AttractionForce.Eades: return AttractEades(diff);
				case AttractionForce.FruchtermanReingold: return AttractFR(diff);
				default: return AttractSpring(diff);
			}
			
		}
		
		public static Vector3D Repulse(Vector3D diff)
		{
			switch(Repulsion)
			{
				case RepulsionForce.Eades: return RepulseEades(diff);
				case RepulsionForce.FruchtermanReingold: return RepulseFR(diff);
				default: return RepulseAttractor(diff);
			}
		}
		
		//Eades
		private static Vector3D AttractEades(Vector3D diff)
		{
			var l = Math.Abs(diff|diff);
			diff /= l;
			
			return diff * ConstantsEades.x * Math.Log(l/ConstantsEades.y);
		}
		
		private static Vector3D RepulseEades(Vector3D diff)
		{
			var l = Math.Abs(diff|diff);
			diff /= l;
			return diff * Math.Min(ConstantsEades.z / l, 10);
		}
		
		//Fruchterman-Reingold
		private static Vector3D AttractFR(Vector3D diff)
		{
			var l = !diff;
			diff /= l;
			
			return diff * ((l*l)/ConstantsFR.x);
		}
		
		private static Vector3D RepulseFR(Vector3D diff)
		{
			var l = !diff;
			diff /= l;
			return diff * ((ConstantsFR.y*ConstantsFR.y) / l);
		}
		
		//SpringAttractor
		private static Vector3D AttractSpring(Vector3D diff)
		{
			var l = !diff;
			diff /= l;
			
			return diff * (l - ConstantsSA.x) * 0.5;
		}
		
		private static Vector3D RepulseAttractor(Vector3D diff)
		{
		
			var l = !diff;
			
			if(l > ConstantsSA.y) return new Vector3D(0);
			
			diff /= l;
			var s = Math.Pow(l/ConstantsSA.y, 1/ConstantsSA.z);
			var f = s * 9 * ConstantsSA.w * ( 1/(s+1) + ((s-3)/4) ) / l;
			return diff * f;
		}
	}
}
