
#ifndef BEZIER_HS_PARTITION
#define BEZIER_HS_PARTITION "integer"
#endif // BEZIER_HS_PARTITION

#define INPUT_PATCH_SIZE 9

#define OUTPUT_PATCH_SIZE 9


    float4x4 g_mViewProjection : WORLDVIEWPROJECTION;
    float3 g_vCameraPosWorld;
    float  g_fTessellationFactor;
StructuredBuffer<float4> cols;
float2 countxy = 1;

//--------------------------------------------------------------------------------------
// Vertex shader section
//--------------------------------------------------------------------------------------
struct VS_CONTROL_POINT_INPUT
{
	uint ii : SV_InstanceID;
    float3 vPosition        : POSITION;
};

struct VS_CONTROL_POINT_OUTPUT
{
    float3 vPosition        : POSITION;
	uint ii : TEXCOORD0;
};


VS_CONTROL_POINT_OUTPUT BezierVS( VS_CONTROL_POINT_INPUT Input )
{
    VS_CONTROL_POINT_OUTPUT Output;

    Output.vPosition = Input.vPosition;
	Output.ii = Input.ii;

    return Output;
}

//--------------------------------------------------------------------------------------
// Constant data function for the BezierHS.  This is executed once per patch.
//--------------------------------------------------------------------------------------
struct HS_CONSTANT_DATA_OUTPUT
{
    float Edges[4]             : SV_TessFactor;
    float Inside[2]            : SV_InsideTessFactor;
};

struct HS_OUTPUT
{
    float3 vPosition           : BEZIERPOS;
	uint ii : TEXCOORD0;
};

HS_CONSTANT_DATA_OUTPUT BezierConstantHS( InputPatch<VS_CONTROL_POINT_OUTPUT, INPUT_PATCH_SIZE> ip,
                                          uint PatchID : SV_PrimitiveID )
{    
    HS_CONSTANT_DATA_OUTPUT Output;

    float TessAmount = g_fTessellationFactor;

    Output.Edges[0] = Output.Edges[1] = Output.Edges[2] = Output.Edges[3] = TessAmount;
    Output.Inside[0] = Output.Inside[1] = TessAmount;

    return Output;
}

[domain("quad")]
[partitioning(BEZIER_HS_PARTITION)]
[outputtopology("triangle_ccw")]
[outputcontrolpoints(OUTPUT_PATCH_SIZE)]
[patchconstantfunc("BezierConstantHS")]
HS_OUTPUT BezierHS( InputPatch<VS_CONTROL_POINT_OUTPUT, INPUT_PATCH_SIZE> p, 
                    uint i : SV_OutputControlPointID,
                    uint PatchID : SV_PrimitiveID )
{
    HS_OUTPUT Output;
    Output.vPosition = p[i].vPosition;
	Output.ii = p[i].ii;
    return Output;
}

struct DS_OUTPUT
{
    float4 vPosition        : SV_POSITION;
    float3 vWorldPos        : WORLDPOS;
    float3 vNormal            : NORMAL;
	float4 col : COLOR;
};

float3 BernsteinBasis(float t)
{
    float invT = 1.0f - t;
    return float3( invT * invT,2.0f * t * invT,t * t);
}

float3 EvaluateBezier( const OutputPatch<HS_OUTPUT, OUTPUT_PATCH_SIZE> bezpatch,
                       float3 BasisU,
                       float3 BasisV )
{
    float3 Value = float3(0,0,0);
    Value  = BasisV.x * ( bezpatch[0].vPosition * BasisU.x + bezpatch[1].vPosition * BasisU.y + bezpatch[2].vPosition * BasisU.z);
    Value += BasisV.y * ( bezpatch[3].vPosition * BasisU.x + bezpatch[4].vPosition * BasisU.y + bezpatch[5].vPosition * BasisU.z);
    Value += BasisV.z * ( bezpatch[6].vPosition * BasisU.x + bezpatch[7].vPosition * BasisU.y + bezpatch[8].vPosition * BasisU.z);

    return Value;
}


[domain("quad")]
DS_OUTPUT BezierDS( HS_CONSTANT_DATA_OUTPUT input, 
                    float2 UV : SV_DomainLocation,
                    const OutputPatch<HS_OUTPUT, OUTPUT_PATCH_SIZE> bezpatch )
{
	uint ii = bezpatch[0].ii;
	
	float colid = ii % countxy.x;
	
	UV.x /= countxy.x;
	UV.x += colid / countxy.x; //(UV.x / countxy) * ii;
	
	int rowid = ii / countxy.x;
	UV.y /= countxy.y;
	UV.y += rowid / countxy.y;

    float3 BasisU = BernsteinBasis( UV.x );
    float3 BasisV = BernsteinBasis( UV.y );

    float3 WorldPos = EvaluateBezier( bezpatch, BasisU, BasisV );
	
    DS_OUTPUT Output;
    Output.vPosition = mul( float4(WorldPos,1), g_mViewProjection );
    Output.vWorldPos = WorldPos;
    Output.vNormal = 0;
	Output.col = cols[ii];

    return Output;    
}


float4 BezierPS( DS_OUTPUT Input ) : SV_TARGET
{
    float3 N = normalize(Input.vNormal);
    float3 L = normalize(Input.vWorldPos - g_vCameraPosWorld);
    return abs(dot(N, L)) * float4(1, 0, 0, 1);
}

float4 PS( DS_OUTPUT Input ) : SV_TARGET
{
    return Input.col;
}


technique11 Render1D
{
	pass P0
	{
		SetHullShader( CompileShader( hs_5_0, BezierHS() ) );
		SetDomainShader( CompileShader( ds_5_0, BezierDS() ) );
		SetVertexShader( CompileShader( vs_5_0, BezierVS() ) );
		SetPixelShader( CompileShader( ps_5_0, PS() ) );
	}
}