//@author: dottore
//@description: Draws a surface using the data position texture. shading by phong directional
//@tags: 3d surface
//@credits: 
// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD ;        //the models world matrix
float4x4 tV: VIEW ;         //view matrix as set via Renderer (EX9)
float4x4 tVP: VIEWPROJECTION ;
float4x4 tWV: WORLDVIEW ;
float4x4 tWVP: WORLDVIEWPROJECTION ;
float4x4 matVI: VIEWINVERSE ;
float4x4 worldIT: WorldInverseTranspose;

#define NUM_ITERATIONS 16
#define NUM_ITERATIONS_RELIEF1 11
#define NUM_ITERATIONS_RELIEF2 5

float4x4 tTex: TEXTUREMATRIX <string uiname="Texture Transform";>;

//texture
float lAtt0 <String uiname="Light Attenuation 0"; float uimin=0.0;> = 0;
float lAtt1 <String uiname="Light Attenuation 1"; float uimin=0.0;> = 0.3;
float lAtt2 <String uiname="Light Attenuation 2"; float uimin=0.0;> = 0;
float lPower <String uiname="Power"; float uimin=0.0;> = 25.0;     //shininess of specular highlight

float4 lDiff : COLOR <String uiname="Object Color";>  = {0.85, 0.85, 0.85, 1};
texture diffTex <String uiname="Diffuse Texture";>;
sampler diffSamp = sampler_state
{
    Texture   = (diffTex);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};
float4 lAmb  : COLOR <String uiname="Ambient Color";>  = {0.15, 0.15, 0.15, 1};
float4 lEm  : COLOR <String uiname="Emission Color";>  = {0, 0, 0, 1};
texture emTex <String uiname="Emission Texture";>;
sampler emSamp = sampler_state
{
    Texture   = (emTex);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};
float4 lSpec : COLOR <String uiname="Specular Color";> = {0.35, 0.35, 0.35, 1};
float specAmount <String uiname="Specular Amount";> = 1;
texture specMap <String uiname="Specular Map";>;
sampler specSamp = sampler_state
{
    Texture   = (specMap);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};
texture TexBump <string uiname="Bump Map";>;
sampler2D cone_map = sampler_state
{
	Texture = <TexBump>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};
texture TexNorm <string uiname="Normal Map";>;
sampler2D normal_map = sampler_state
{
	Texture = <TexNorm>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};
sampler3D sphere_map = sampler_state
{
	Texture = <TexBump>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};
texture TexDisp <string uiname="Displacement Map";>;
sampler2D disp_map = sampler_state
{
	Texture = <TexDisp>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};
texture TexDispNorm <string uiname="Displacement Normal Map";>;
sampler2D dispNorm_map = sampler_state
{
	Texture = <TexDispNorm>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};
float Displace = 0;
float displaceOffset = 0;

bool isPTxCd <string uiname="Position as Texture Coordinates";> = 0;
float alphatest = 0.5;
float depth = 0;
float posDepth = 0.3;
float normamount = 1;
float bumpOffset = 0;
bool DEPTH_BIAS = false;
bool BORDER_CLAMP = false;
float3 size_source;

float reflectStrength
<
    string UIWidget = "slider";
    float UIMin = 0.0;
    float UIMax = 1.0;
    float UIStep = 0.01;
    string UIName =  "Reflection";
> = 0;

float refractStrength
<
    string UIWidget = "slider";
    float UIMin = 0.0;
    float UIMax = 1.0;
    float UIStep = 0.01;
    string UIName =  "Refraction";
> = 0;

float rrBump
<
    string UIName =  "Bumpiness";
> = 0;
half3 etas
<
    string UIName = "Refraction indices";
> = { 0.80, 0.82, 0.84 };

texture cubeMap : Environment
<
	string ResourceName = "default_reflection.dds";
	string ResourceType = "Cube";
>;

texture fresnelTex : Environment
<
	string ResourceType = "2D";
	string function = "generateFresnelTex";
	
	float2 Dimensions = { 256.0f, 1.0f};
>;

samplerCUBE environmentMapSampler = sampler_state
{
	Texture = <cubeMap>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};

sampler2D fresnelSampler = sampler_state
{
	Texture = <fresnelTex>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
};
texture ReflM <string uiname="Reflection Map";>;
sampler2D reflmap = sampler_state
{
	Texture = <ReflM>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};

// -----------------------------------------------------------------------------
// MISC:
// -----------------------------------------------------------------------------
float3 cone(float2 uv0, float3 eyeVec, sampler2D cone_map)
{
	float3 rayPos;
	float3 rayVec;
	rayPos = float3(uv0, 0.0);
	
	rayVec = normalize(eyeVec);
	rayVec.z = abs(rayVec.z);

	// DerayPosth bias used by Fabio rayPosolicarrayPoso
	if (DEPTH_BIAS)
	{
		float db = 1.0 - rayVec.z;
		db *= db;
		db = 1.0 - db*db;
		rayVec.xy *= db;
	}
	
	rayVec.xy *= depth;

	float dist = length(rayVec.xy);
	
	for( int i=0;i<NUM_ITERATIONS; i++ )
	{
		float4 tex = tex2D(cone_map, rayPos.xy)+bumpOffset;
		float height = saturate(tex.w - rayPos.z);

		float cone_ratio = tex.z*tex.z;
		float stepDist = height * cone_ratio / (cone_ratio + dist);
		rayPos += rayVec * stepDist;
		
	}
	
	return rayPos.xyz;
}

float3 refract2( float3 I, float3 N, float eta, out bool fail )
{
	float IdotN = dot(I, N);
	float k = 1 - eta*eta*(1 - IdotN*IdotN);
//	return k < 0 ? (0,0,0) : eta*I - (eta*IdotN + sqrt(k))*N;
	fail = k < 0;
	return eta*I - (eta*IdotN + sqrt(k))*N;
}

// approximate Fresnel function
float fresnel(float NdotV, float bias, float power)
{
   return bias + (1.0-bias)*pow(1.0 - max(NdotV, 0), power);
}

// function to generate a texture encoding the Fresnel function
float4 generateFresnelTex(float NdotV : POSITION) : COLOR
{
	return fresnel(NdotV, 0.2, 4.0);
}

// -----------------------------------------------------------------------------
// STRUCT:
// -----------------------------------------------------------------------------
struct appdata
{
    float4 PosO: POSITION;
    float3 NormO: NORMAL;
    float2 TexCdO: TEXCOORD0;
    float3 tang: TANGENT;
    float3 bino: BINORMAL;
};

struct vs2ps2
{
    float4 PosWVPp: POSITION;
    float4 PosWVP: TEXCOORD0;
    float4 PosW: TEXCOORD1;
    float3 PosWV : TEXCOORD2;
    float3 NormWV: NORMAL;
    float3 NormW: TEXCOORD3;
    float3 ViewDirWV: TEXCOORD4;
	float2 TexCd: TEXCOORD5;
	float3 eyeVec: TEXCOORD6;
};

// -----------------------------------------------------------------------------
// VERTEXSHADERS:
// -----------------------------------------------------------------------------

// PLACE and DEFORM technique
vs2ps2 VS(appdata In)
{
    //inititalize all fields of output struct with 0
    vs2ps2 Out = (vs2ps2)0;
	
	if(isPTxCd) Out.TexCd = mul(In.PosO, tTex);
	else Out.TexCd = mul(float4(In.TexCdO,0,1), tTex);
	
	float4 dispPos = In.PosO;
	float3 dispNorm = In.NormO;
	float4 pdispPos = In.PosO;
	if(Displace!=0)
	{
		dispPos += tex2Dlod(disp_map, float4(Out.TexCd,0,1)).r * float4(In.NormO,1) * Displace + displaceOffset;
		dispPos.w = 1;
		dispNorm += ((2 * (tex2Dlod(dispNorm_map, float4(Out.TexCd,0,1)))) - 1.0)*-Displace;
	}

    Out.PosW = mul(dispPos, tW);
    Out.PosWV = mul(dispPos, tWV);
    Out.PosWVP = mul(dispPos, tWVP);
    Out.PosWVPp = mul(dispPos, tWVP);
	float3 npos = Out.PosWVP.xyz;

    //normal in view space
    Out.NormWV = normalize(mul(dispNorm, tWV));
    Out.NormW = normalize(mul(dispNorm, tW));
    
    float3x3 tangentMap = float3x3(In.tang, In.bino, dispNorm);
    tangentMap = mul(tangentMap, tW);
	float3 eyeVec = Out.PosW - matVI[3].xyz;	
	Out.eyeVec = mul(tangentMap, eyeVec);

	//Out.ViewDirV = -normalize(mul(Out.PosW, tWV));
    Out.ViewDirWV = -normalize(mul(dispPos, tWV));
	
    return Out;
}

// -----------------------------------------------------------------------------
// MRT STRUCT:
// -----------------------------------------------------------------------------
struct col
{
    float4 lightprop : COLOR0 ;
    float4 ambient : COLOR1 ;
    float4 emission : COLOR2 ;
    float4 specular : COLOR3 ;
};


// -----------------------------------------------------------------------------
// PIXELSHADERS:
// -----------------------------------------------------------------------------

col PS1(vs2ps2 In): COLOR
{
    col c;
	
	c.lightprop.r = lAtt0;
	c.lightprop.g = lAtt1;
	c.lightprop.b = lPower;
	
	c.ambient = lAmb;
	
	float3 normb = In.NormWV;
	float3 normWb = In.NormW;
	float2 itexcd = In.TexCd.xy;
	itexcd.x *= -1;
	float3 uvb = cone(itexcd, In.eyeVec, cone_map);
	if(depth!=0) normb += ((2 * (tex2D(normal_map,itexcd))) - 1.0)*-depth;
	float3 posWb = In.PosW;
	if(depth!=0) posWb += In.NormW * uvb.z * (-1*pow(depth,.5)) * posDepth;
	float3 posb = In.PosWV;
	if(depth!=0) posb += In.NormWV * uvb.z * (-1*pow(depth,.5)) * posDepth;
	float3 ViewDirWV = -normalize(posb);
	
	float alphat = 1;
	float alphatt = tex2D(diffSamp, uvb.xy).a * lDiff.a;
	alphat = alphatt;
	if(alphatest!=0) alphat = lerp(alphatt, (alphatt>=alphatest), min(alphatest*10,1));
	
	c.ambient.a = alphat;
	c.lightprop.a = alphat;

	c.emission.rgb = lEm.rgb * tex2D(emSamp, uvb.xy).rgb;
	
	c.specular.rgb = lSpec.rgb * tex2D(specSamp, uvb.xy).rgb * specAmount;
	c.specular.a = alphat;
	
	//ReflectRefract
	if((reflectStrength!=0) || (refractStrength!=0))
	{
	float3 rrnorm = ((2 * (tex2D(normal_map,itexcd))) - 1.0)*-rrBump;
	half3 rrN = normalize(mul(In.NormWV+rrnorm,matVI));
	float3 rrV = normalize(matVI[3].xyz - (In.PosW.xyz+rrN*uvb.z*-1*rrBump*posDepth));
    half3 rrR = reflect(-rrV, rrN);
    half4 reflColor = texCUBE(environmentMapSampler, rrR);
	half fresnel = tex2D(fresnelSampler, dot(rrN, rrV));
	const half4 colors[3] =
        {
    	{ 1, 0, 0, 1 },
    	{ 0, 1, 0, 1 },
    	{ 0, 0, 1, 1 },
	};
	half4 transColor = 0;
  	bool fail = false;
    for(int i=0; i<3; i++) {
    	half3 rrT = refract2(-rrV, rrN, etas[i], fail);
    	transColor += texCUBE(environmentMapSampler, rrT)* colors[i];
	}
	c.emission += reflColor*reflectStrength*2*fresnel*tex2D(reflmap, float2(uvb.x*-1, uvb.y));
	c.emission += transColor*refractStrength*(1-fresnel);
	}
	
	c.emission.a = alphat;

    return c;
}

// -----------------------------------------------------------------------------
// TECHNIQUES:
// -----------------------------------------------------------------------------

technique Phong_PointLight
{
    pass P0
    {
        //Wrap0 = U;  // useful when mesh is round like a sphere
        VertexShader = compile vs_3_0 VS();
        PixelShader  = compile ps_3_0 PS1();
    }
}