#region usings
using System;
using System.ComponentModel.Composition;
using System.Text;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using System.Globalization;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;
using VVVV.Utils.Streams;
using VVVV.Core.Logging;

using TPCANHandle = System.Byte;
#endregion usings

namespace VVVV.Nodes
{
	#region PluginInfo
	[PluginInfo(Name = "CanBus",
            Category = "Devices",
            Help = "Reads and Sends CAN BUS messages over the Peak Can Adapter",
            Tags = "CANBUS, bus",
            Author = "ethermammoth",
            AutoEvaluate = false)]
	#endregion PluginInfo
	public class DevicesCanBusNode : IPluginEvaluate
	{
		#region fields & pins
		[Input("Connect", IsSingle = true)]
		ISpread<bool> FConnect;
		
		[Input("Send")]
		IDiffSpread<bool> FSend;
		
		[Input("Init Buffer", IsSingle = true)]
		ISpread<bool> FInitBuffer;
		
		[Input("Message ID")]
		IDiffSpread<string> FSendMessageID;
		
		[Input("Message Count")]
		IDiffSpread<int> FSendMessageCount;
		
		[Input("Message")]
		IDiffSpread<string> FSendMessage;
		
		[Input("Message Length")]
		IDiffSpread<int> FSendMessageLength;
		
		[Input("Send Message Every Milli")]
		IDiffSpread<int> FSendMessageTime;
		
		[Input("BUS Type", IsSingle = true, EnumName = "CanBusType")]
        IDiffSpread<EnumEntry> FBusType;
		
		[Input("Baud Rate", IsSingle = true, EnumName = "CanBusBaudRate")]
        IDiffSpread<EnumEntry> FBaudRate;

		[Output("Message ID")]
		ISpread<string> FMessageID;
		
		[Output("Message TimeStamp")]
		ISpread<string> FMessageTimeStamp;
		
		[Output("Message Length")]
		ISpread<int> FMessageLength;
		
		[Output("Message Data")]
		ISpread<string> FMessageData;
		
		[Output("Message Count")]
		ISpread<int> FMessageCount;
		
		[Output("Actual Send Time")]
		ISpread<int> FTimeUsedSending;
		
		[Output("Error Msg")]
		ISpread<string> FErrorMsg;

		[Import()]
		ILogger FLogger;
		
		[ImportingConstructor]
        public DevicesCanBusNode()
		{
            var s = new string[] { "1 MBit/s", "800 KBit/s", "500 kBit/s", "250 kBit/s", "125 kBit/s",
			"100 kBit/s", "95,238 KBit/s", "83,333 KBit/s", "50 kBit/s", "47,619 KBit/s", "33,333 KBit/s",
			"20 kBit/s", "10 kBit/s", "5 kBit/s" };
		    EnumManager.UpdateEnum("CanBusBaudRate", "500 kBit/s", s);
			var m = new string[] { "USB-BUS-1", "USB-BUS-2", "USB-BUS-3", "USB-BUS-4", "USB-BUS-5",
			"USB-BUS-6", "USB-BUS-7", "USB-BUS-8" };
			EnumManager.UpdateEnum("CanBusType", "USB-BUS-1", m);
		}
		
		private bool hasInitialized = false;
		private TPCANHandle m_PcanHandle;
		private TPCANBaudrate m_Baudrate;
		private System.Collections.ArrayList m_LastMsgsList;
		//Buffer for sending messages (to avoid conversion on every frame)
		private int bufferCount;
		private BufferMessage[] msgBuffer;
		private bool bufferInitialized = false;
		//Timer for sending messages with millis. accuracy
		private Stopwatch timer;
		private long frametime;
		private long prevframetime;
		#endregion fields & pins
		

		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			if(FConnect[0] && !hasInitialized)
			{
				Initialize();
				//start timer
				timer = new Stopwatch();
				timer.Reset();
				hasInitialized = true;
			}
			
			if(!FConnect[0] && hasInitialized)
			{
				Unitialize();
				hasInitialized = false;
			}
			
			//init message buffer
			if(FInitBuffer[0])
			{
				FErrorMsg[0] = "";
				//first check if lengths are matching (otherwise give warning to user)
				int countv = 0;
				for(int x=0; x<FSendMessageCount.SliceCount; x++)
				{
					countv += FSendMessageCount[x];	
				}
				
				if(countv != FSendMessage.SliceCount)
				{
					FErrorMsg[0] = "Error Mismatching length in message count and data spread";
					return;
				}
				
				InitSendMessageBuffer();
				FTimeUsedSending.SliceCount = bufferCount;
				bufferInitialized = true;
			}			
			
			if(FConnect[0] && hasInitialized)
			{
				ReadMessages();
				ReadSpreadMessages();
			}
			
			//This is for sending Cyclic buffered messages
			//(in order to increase performance and avoid conversion)
			if(FConnect[0] && hasInitialized && FSend[0] && bufferInitialized)
			{
				if(FSend.IsChanged)
				{
					//start the timer
					timer.Reset();
					timer.Start();
				}
				//update frame timer
				frametime = timer.ElapsedMilliseconds - prevframetime;
				prevframetime = timer.ElapsedMilliseconds;
				//loop through buffer and send messages when the cycle time has arrived
				//internally the buffer updates the index and send times
				for(int x=0; x<bufferCount; x++)
				{
					if( msgBuffer[x].TimeToSend( timer.ElapsedMilliseconds, frametime ) )
					{
						SendMessage( msgBuffer[x].GetSendMessage() );
						FTimeUsedSending[x] = msgBuffer[x].GetSendTime();
					}
				}				
			}
			
			//This is for sending imidiate (one time messages)
			//Sends everything connected to the input
			//Uses the cycle time input to determine the aktuall sending
			// 0 = No Send 1 = Send
			if(FConnect[0] && hasInitialized)
			{
				for(int x=0; x<FSendMessageID.SliceCount; x++)
				{
					if(FSendMessageTime[x] == 1)
					{
						TPCANMsg msg = ConvertMessage(FSendMessageID[x], FSendMessage[x], FSendMessageLength[x]);
						SendMessage(msg);
					}
				}
			}
			
			//update baut when it changes
			if(FBaudRate.IsChanged)
			{
				SetBaudRate();
				//when running re-init
				if(FConnect[0] && hasInitialized)
				{
					Unitialize();
					Initialize();
				}
			}
			
			//update type when it changes
			if(FBusType.IsChanged)
			{
				SetType();
				//when running re-init
				if(FConnect[0] && hasInitialized)
				{
					Unitialize();
					Initialize();
				}
			}
			
		}
		
		private void InitSendMessageBuffer()
		{
			//init message buffer
			bufferCount = FSendMessageCount.SliceCount;
			msgBuffer = new BufferMessage[bufferCount];
			//fill buffer with data
			int dataIndex = 0;
			for(int i=0; i<bufferCount; i++)
			{
				msgBuffer[i] = new BufferMessage(FSendMessageCount[i], FSendMessageTime[i]);
				//fill buffer object with data
				for(int x=0; x<FSendMessageCount[i]; x++)
				{
					TPCANMsg msg = ConvertMessage(FSendMessageID[i], FSendMessage[dataIndex], FSendMessageLength[i]);
					
					if(!msgBuffer[i].AddMessage(x, msg))
					{
						FErrorMsg[0] = "Error Building Buffer - Index out of Range";
						break;						
					}
					dataIndex++;
				}					
			}
		}		
		
		private bool OnlyHexInString(string test)
		{
		    // For C-style hex notation (0xFF) you can use @"\A\b(0[xX])?[0-9a-fA-F]+\b\Z"
		    return System.Text.RegularExpressions.Regex.IsMatch(test, @"\A\b[0-9a-fA-F]+\b\Z");
		}
		
		private string ConvertToAscii(string hexString)
		{
			string cnv = "";
			hexString = Regex.Replace(hexString, @"\s", "" );	//whitespace
			for (int j = 0; j < hexString.Length; j+=2)
				cnv += (char)Int32.Parse(hexString.Substring(j,2), NumberStyles.AllowHexSpecifier);
			
			return cnv;
		}
		
		private byte[] ConvertHexStringToByteArray(string hexString)
		{
		    if (hexString.Length % 2 != 0)
		    {
		        //throw new ArgumentException(String.Format(CultureInfo.InvariantCulture, "The binary key cannot have an odd number of digits: {0}", hexString));
		    	FErrorMsg[0] = "HEX HAS BAD FORMAT (LENGTH)";
		    	return new byte[8];
		    }
			
			byte[] HexAsBytes = new byte[8];
		    for (int index = 0; index < hexString.Length / 2; index++)
		    {
		        string byteValue = hexString.Substring(index * 2, 2);
		        HexAsBytes[index] = byte.Parse(byteValue, NumberStyles.HexNumber, CultureInfo.InvariantCulture);
		    }
		
		    return HexAsBytes; 
		}
		
		//Message Conversion
		private TPCANMsg ConvertMessage(string id, string message, int length)
		{
			message = Regex.Replace(message, @"\s", "" );	//whitespace
			if( OnlyHexInString( message ) )
			{
				byte[] testData = ConvertHexStringToByteArray(message);
				
				TPCANMsg CANMsg = new TPCANMsg();
				CANMsg.DATA = testData;
				try
				{
					CANMsg.ID = Convert.ToUInt32(id, 16);
				}catch(Exception e)
				{
					FErrorMsg[0] = e.ToString();
				}
				CANMsg.LEN = Convert.ToByte(length);
				CANMsg.MSGTYPE = TPCANMessageType.PCAN_MESSAGE_STANDARD;
				return CANMsg;
			}else{
				FErrorMsg[0] = "Message is not valid Hex Notation " + id;
				FLogger.Log(LogType.Error, "Bad HEX: " + id);
				return new TPCANMsg();
			}
		}
		
		//Send message
		private void SendMessage(TPCANMsg message)
		{
			TPCANStatus stsResult;
			
			stsResult = PCANBasic.Write(m_PcanHandle, ref message);
			if(stsResult != TPCANStatus.PCAN_ERROR_OK)
			{
				FLogger.Log(LogType.Error, "CanBus: Send Error! " + stsResult.ToString());
			}
		}
		
		//Converts messages to vvvv readable spreads
		private void ReadSpreadMessages()
		{
			int numberOfMessages = m_LastMsgsList.Count;
			FMessageID.SliceCount = numberOfMessages;
			FMessageLength.SliceCount = numberOfMessages;
			FMessageData.SliceCount = numberOfMessages;
			FMessageTimeStamp.SliceCount = numberOfMessages;
			FMessageCount.SliceCount = numberOfMessages;
			
			int i=0;
			foreach(MessageStatus msg in m_LastMsgsList)
			{
				FMessageID[i] = msg.IdString;
				FMessageLength[i] = msg.CANMsg.LEN;
				FMessageData[i] = msg.DataString;
				FMessageTimeStamp[i] = msg.TimeString;
				FMessageCount[i] = msg.Count;
				i++;
			}
		}
		
		private void ReadMessages()
		{
			TPCANMsg CANMsg;
            TPCANTimestamp CANTimeStamp;
            TPCANStatus stsResult;

            // We read at least one time the queue looking for messages.
            // If a message is found, we look again trying to find more.
            // If the queue is empty or an error occurr, we get out from
            // the dowhile statement.
            //			
            do
            {
                // We execute the "Read" function of the PCANBasic                
                //
                stsResult = PCANBasic.Read(m_PcanHandle, out CANMsg, out CANTimeStamp);

                // A message was received
                // We process the message(s)
                //
                if (stsResult == TPCANStatus.PCAN_ERROR_OK)
            	{
            		ProcessMessage(CANMsg, CANTimeStamp);
            	}

            } while (!Convert.ToBoolean(stsResult & TPCANStatus.PCAN_ERROR_QRCVEMPTY));
		}
		
		private void ProcessMessage(TPCANMsg theMsg, TPCANTimestamp itsTimeStamp)
		{
			foreach (MessageStatus msg in m_LastMsgsList)
            {
                if ((msg.CANMsg.ID == theMsg.ID) && (msg.CANMsg.MSGTYPE == theMsg.MSGTYPE))
                {
                    // Modify the message and exit
                    //
                    msg.Update(theMsg, itsTimeStamp);
                    return;
                }
            }
            // Message not found. It will created
            //
            InsertMsgEntry(theMsg, itsTimeStamp);
		}
		
		private void InsertMsgEntry(TPCANMsg newMsg, TPCANTimestamp timeStamp)
        {
            MessageStatus msgStsCurrentMsg;
			int msgCount = m_LastMsgsList.Count;
            lock (m_LastMsgsList.SyncRoot)
            {
                msgStsCurrentMsg = new MessageStatus(newMsg, timeStamp, msgCount);
                m_LastMsgsList.Add(msgStsCurrentMsg);
            }
        }
		
		private void Initialize()
		{
			FLogger.Log(LogType.Debug, "CanBus: Trying to initialize CANBUS");
			TPCANStatus tStatus;
			m_LastMsgsList = new System.Collections.ArrayList();
			
			if(m_PcanHandle != null && m_Baudrate != null)
			{
				tStatus = PCANBasic.Initialize(m_PcanHandle, m_Baudrate);
			}else{
				tStatus = PCANBasic.Initialize(PCANBasic.PCAN_USBBUS1, TPCANBaudrate.PCAN_BAUD_500K);
			}
			
			if(tStatus != TPCANStatus.PCAN_ERROR_OK)
			{
				//some error ocured
				FErrorMsg[0] = "Error Init USB-C1";
				return;
			}
			FLogger.Log(LogType.Debug, "CanBus: Init success!");
		}
		
		private void Unitialize()
		{
			FLogger.Log(LogType.Debug, "CanBus: Uninitializing!");
			PCANBasic.Uninitialize(m_PcanHandle);
			FLogger.Log(LogType.Debug, "CanBus: Uninitializing success!");
		}
		
		private void SetBaudRate()
		{
			FLogger.Log(LogType.Debug, "CanBus: Changing Baud Rate!");
			switch(FBaudRate[0].Index)
			{
				case 0:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_1M;
				break;
				case 1:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_800K;
				break;
				case 2:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_500K;
				break;
				case 3:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_250K;
				break;
				case 4:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_125K;
				break;
				case 5:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_100K;
				break;
				case 6:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_95K;
				break;
				case 7:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_83K;
				break;
				case 8:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_50K;
				break;
				case 9:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_47K;
				break;
				case 10:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_33K;
				break;
				case 11:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_20K;
				break;
				case 12:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_10K;
				break;
				case 13:
					m_Baudrate = TPCANBaudrate.PCAN_BAUD_5K;
				break;
			}
			FLogger.Log(LogType.Debug, "CanBus: Baud Rate changed to ->" + m_Baudrate.ToString());
		}
		
		private void SetType()
		{
			FLogger.Log(LogType.Debug, "CanBus: Changing BUS Type!");
			switch(FBusType[0].Index)
			{
				case 0:
					m_PcanHandle = PCANBasic.PCAN_USBBUS1;
				break;
				case 1:
					m_PcanHandle = PCANBasic.PCAN_USBBUS2;
				break;
				case 2:
					m_PcanHandle = PCANBasic.PCAN_USBBUS3;
				break;
				case 3:
					m_PcanHandle = PCANBasic.PCAN_USBBUS4;
				break;
				case 4:
					m_PcanHandle = PCANBasic.PCAN_USBBUS5;
				break;
				case 5:
					m_PcanHandle = PCANBasic.PCAN_USBBUS6;
				break;
				case 6:
					m_PcanHandle = PCANBasic.PCAN_USBBUS7;
				break;
				case 7:
					m_PcanHandle = PCANBasic.PCAN_USBBUS8;
				break;
			}
			FLogger.Log(LogType.Debug, "CanBus: BUS type changed to ->" + m_PcanHandle.ToString());
		}
	}
	
	public class BufferMessage
	{
		private TPCANMsg[] messages;
		private int messageCount; //message Count (messages length)
		private int currentIndex;
		private long lastTimeSend;
		private long cycleTime;
		private long sendTime;
		
		public BufferMessage(int count, long ctime)
		{
			//create the buffer array
			messages = new TPCANMsg[count];
			messageCount = count;
			lastTimeSend = 0;
			currentIndex = 0;
			sendTime = 0;
			cycleTime = ctime;
		}
		
		public bool AddMessage(int index, TPCANMsg msg)
		{
			try
			{
				messages[index] = msg;
			}
			catch (System.IndexOutOfRangeException e) 
			{
				System.Console.WriteLine(e.Message);
				return false;
			}
			return true;
		}
		
		public bool TimeToSend(long currentTime, long frameTime)
		{
			if(lastTimeSend == 0)
			{
				lastTimeSend = currentTime;
				return true;
			}
			
			long timeUsed = currentTime - lastTimeSend;
			long timeAhead = Math.Abs(cycleTime - (timeUsed + frameTime));
			long timeNow = Math.Abs(cycleTime - timeUsed);
			long addTime = 0;
			if(timeAhead > timeNow)
				addTime += frameTime;
			
			if(timeUsed + addTime >= cycleTime)
			{
				sendTime = timeUsed;
				lastTimeSend = currentTime;
				return true;
			}
			return false;
		}
		
		public int GetSendTime()
		{
			return (int)sendTime;
		}		
		
		public TPCANMsg GetSendMessage()
		{
			int index = currentIndex;
			if(currentIndex + 1 < messageCount)
				currentIndex++;
			else
				currentIndex = 0;
			
			return messages[index];
		}
		
	}
	
	public class MessageStatus
    {
        private TPCANMsg m_Msg;
        private TPCANTimestamp m_TimeStamp;
        private TPCANTimestamp m_oldTimeStamp;
        private int m_iIndex;
        private int m_Count;
        private bool m_bShowPeriod;
        private bool m_bWasChanged;

        public MessageStatus(TPCANMsg canMsg, TPCANTimestamp canTimestamp, int listIndex)
        {
            m_Msg = canMsg;
            m_TimeStamp = canTimestamp;
            m_oldTimeStamp = canTimestamp;
            m_iIndex = listIndex;
            m_Count = 1;
            m_bShowPeriod = true;
            m_bWasChanged = false;
        }

        public void Update(TPCANMsg canMsg, TPCANTimestamp canTimestamp)
        {
            m_Msg = canMsg;
            m_oldTimeStamp = m_TimeStamp;
            m_TimeStamp = canTimestamp;
            m_bWasChanged = true;
            m_Count += 1;
        }

        public TPCANMsg CANMsg
        {
            get { return m_Msg; }
        }

        public TPCANTimestamp Timestamp
        {
            get { return m_TimeStamp; }
        }

        public int Position
        {
            get { return m_iIndex; }
        }

        public string TypeString
        {
            get { return GetMsgTypeString(); }
        }

        public string IdString
        {
            get { return GetIdString(); }
        }

        public string DataString
        {
            get { return GetDataString(); }
        }

        public int Count
        {
            get { return m_Count; }
        }

        public bool ShowingPeriod
        {
            get { return m_bShowPeriod; }
            set 
            {
                if (m_bShowPeriod ^ value)
                {
                    m_bShowPeriod = value;
                    m_bWasChanged = true;
                }
            }
        }

        public bool MarkedAsUpdated
        {
            get { return m_bWasChanged; }
            set { m_bWasChanged = value; }
        }

        public string TimeString
        {
            get { return GetTimeString(); }
        }

        private string GetTimeString()
        {
            double fTime;

            fTime = m_TimeStamp.millis + (m_TimeStamp.micros / 1000.0);
            if (m_bShowPeriod)
                fTime -= (m_oldTimeStamp.millis + (m_oldTimeStamp.micros / 1000.0));
            return fTime.ToString("F1");                
        }

        private string GetDataString()
        {
            string strTemp;

            strTemp = "";

            if ((m_Msg.MSGTYPE & TPCANMessageType.PCAN_MESSAGE_RTR) == TPCANMessageType.PCAN_MESSAGE_RTR)
                return "Remote Request";
            else
                for (int i = 0; i < m_Msg.LEN; i++)
                    strTemp += string.Format("{0:X2} ", m_Msg.DATA[i]);

            return strTemp;
        }

        private string GetIdString()
        {
            // We format the ID of the message and show it
            //
            if ((m_Msg.MSGTYPE & TPCANMessageType.PCAN_MESSAGE_EXTENDED) == TPCANMessageType.PCAN_MESSAGE_EXTENDED)
                return string.Format("{0:X8}h", m_Msg.ID);
            else
                return string.Format("{0:X3}h", m_Msg.ID);
        }

        private string GetMsgTypeString()
        {
            string strTemp;

            if ((m_Msg.MSGTYPE & TPCANMessageType.PCAN_MESSAGE_EXTENDED) == TPCANMessageType.PCAN_MESSAGE_EXTENDED)
                strTemp = "EXTENDED";
            else
                strTemp = "STANDARD";

            if ((m_Msg.MSGTYPE & TPCANMessageType.PCAN_MESSAGE_RTR) == TPCANMessageType.PCAN_MESSAGE_RTR)
                strTemp += "/RTR";

            return strTemp;
        }

    }	
	
	
	#region Enumerations
    /// <summary>
    /// Represents a PCAN status/error code
    /// </summary>
    [Flags]
    public enum TPCANStatus : uint
    {
        /// <summary>
        /// No error
        /// </summary>
        PCAN_ERROR_OK           = 0x00000,
        /// <summary>
        /// Transmit buffer in CAN controller is full
        /// </summary>
        PCAN_ERROR_XMTFULL      = 0x00001,
        /// <summary>
        /// CAN controller was read too late
        /// </summary>
        PCAN_ERROR_OVERRUN      = 0x00002,  
        /// <summary>
        /// Bus error: an error counter reached the 'light' limit
        /// </summary>
        PCAN_ERROR_BUSLIGHT     = 0x00004,  
        /// <summary>
        /// Bus error: an error counter reached the 'heavy' limit
        /// </summary>
        PCAN_ERROR_BUSHEAVY     = 0x00008,  
        /// <summary>
        /// Bus error: the CAN controller is in bus-off state
        /// </summary>
        PCAN_ERROR_BUSOFF       = 0x00010,  
        /// <summary>
        /// Mask for all bus errors
        /// </summary>
        PCAN_ERROR_ANYBUSERR    = (PCAN_ERROR_BUSLIGHT | PCAN_ERROR_BUSHEAVY | PCAN_ERROR_BUSOFF),
        /// <summary>
        /// Receive queue is empty
        /// </summary>
        PCAN_ERROR_QRCVEMPTY    = 0x00020,  
        /// <summary>
        /// Receive queue was read too late
        /// </summary>
        PCAN_ERROR_QOVERRUN     = 0x00040,  
        /// <summary>
        /// Transmit queue is full
        /// </summary>
        PCAN_ERROR_QXMTFULL     = 0x00080,  
        /// <summary>
        /// Test of the CAN controller hardware registers failed (no hardware found)
        /// </summary>
        PCAN_ERROR_REGTEST      = 0x00100,
        /// <summary>
        /// Driver not loaded
        /// </summary>
        PCAN_ERROR_NODRIVER     = 0x00200,
        /// <summary>
        /// Hardware already in use by a Net
        /// </summary>
        PCAN_ERROR_HWINUSE      = 0x00400,
        /// <summary>
        /// A Client is already connected to the Net
        /// </summary>
        PCAN_ERROR_NETINUSE     = 0x00800,
        /// <summary>
        /// Hardware handle is invalid
        /// </summary>
        PCAN_ERROR_ILLHW        = 0x01400,
        /// <summary>
        /// Net handle is invalid
        /// </summary>
        PCAN_ERROR_ILLNET       = 0x01800,
        /// <summary>
        /// Client handle is invalid
        /// </summary>
        PCAN_ERROR_ILLCLIENT    = 0x01C00,
        /// <summary>
        /// Mask for all handle errors
        /// </summary>
        PCAN_ERROR_ILLHANDLE    = (PCAN_ERROR_ILLHW | PCAN_ERROR_ILLNET | PCAN_ERROR_ILLCLIENT),
        /// <summary>
        /// Resource (FIFO, Client, timeout) cannot be created
        /// </summary>
        PCAN_ERROR_RESOURCE     = 0x02000,
        /// <summary>
        /// Invalid parameter
        /// </summary>
        PCAN_ERROR_ILLPARAMTYPE = 0x04000,
        /// <summary>
        /// Invalid parameter value
        /// </summary>
        PCAN_ERROR_ILLPARAMVAL  = 0x08000,
        /// <summary>
        /// Unknow error
        /// </summary>
        PCAN_ERROR_UNKNOWN      = 0x10000,
        /// <summary>
        /// Invalid data, function, or action.
        /// </summary>
        PCAN_ERROR_ILLDATA      = 0x20000,
        /// <summary>
        /// Channel is not initialized
        /// </summary>
        PCAN_ERROR_INITIALIZE   = 0x40000,
    }

    /// <summary>
    /// Represents a PCAN device
    /// </summary>
    public enum TPCANDevice : byte
    {
        /// <summary>
        /// Undefined, unknown or not selected PCAN device value
        /// </summary>
        PCAN_NONE = 0,
        /// <summary>
        /// PCAN Non-Plug&Play devices. NOT USED WITHIN PCAN-Basic API
        /// </summary>
        PCAN_PEAKCAN = 1,
        /// <summary>
        /// PCAN-ISA, PCAN-PC/104, and PCAN-PC/104-Plus
        /// </summary>
        PCAN_ISA = 2,
        /// <summary>
        /// PCAN-Dongle
        /// </summary>
        PCAN_DNG = 3,
        /// <summary>
        /// PCAN-PCI, PCAN-cPCI, PCAN-miniPCI, and PCAN-PCI Express
        /// </summary>
        PCAN_PCI = 4,
        /// <summary>
        /// PCAN-USB and PCAN-USB Pro
        /// </summary>
        PCAN_USB = 5,
        /// <summary>
        /// PCAN-PC Card
        /// </summary>
        PCAN_PCC = 6
    }

    /// <summary>
    /// Represents a PCAN parameter to be read or set
    /// </summary>
    public enum TPCANParameter : byte
    {
        /// <summary>
        /// PCAN-USB device number parameter
        /// </summary>
        PCAN_DEVICE_NUMBER       = 1,
        /// <summary>
        /// PCAN-PC Card 5-Volt power parameter
        /// </summary>
        PCAN_5VOLTS_POWER        = 2,
        /// <summary>
        /// PCAN receive event handler parameter
        /// </summary>
        PCAN_RECEIVE_EVENT       = 3,
        /// <summary>
        /// PCAN message filter parameter
        /// </summary>
        PCAN_MESSAGE_FILTER      = 4,
        /// <summary>
        /// PCAN-Basic API version parameter
        /// </summary>
        PCAN_API_VERSION         = 5,
        /// <summary>
        /// PCAN device channel version parameter
        /// </summary>
        PCAN_CHANNEL_VERSION     = 6,
        /// <summary>
        /// PCAN Reset-On-Busoff parameter
        /// </summary>
        PCAN_BUSOFF_AUTORESET    = 7,
        /// <summary>
        /// PCAN Listen-Only parameter
        /// </summary>
        PCAN_LISTEN_ONLY         = 8,
        /// <summary>
        /// Directory path for trace files
        /// </summary>
        PCAN_LOG_LOCATION        = 9,
        /// <summary>
        /// Debug-Trace activation status
        /// </summary>
        PCAN_LOG_STATUS          = 10,
        /// <summary>
        /// Configuration of the debugged information (LOG_FUNCTION_***)
        /// </summary>
        PCAN_LOG_CONFIGURE       = 11,
        /// <summary>
        /// Custom insertion of text into the log file
        /// </summary>
        PCAN_LOG_TEXT            = 12,
        /// <summary>
        /// Availability status of a PCAN-Channel
        /// </summary>
        PCAN_CHANNEL_CONDITION   = 13,
        /// <summary>
        /// PCAN hardware name parameter
        /// </summary>
        PCAN_HARDWARE_NAME       = 14,
        /// <summary>
        /// Message reception status of a PCAN-Channel
        /// </summary>
        PCAN_RECEIVE_STATUS      = 15,
        /// <summary>
        /// CAN-Controller number of a PCAN-Channel
        /// </summary>
        PCAN_CONTROLLER_NUMBER   = 16,
    }

    /// <summary>
    /// Represents the type of a PCAN message
    /// </summary>
    [Flags]
    public enum TPCANMessageType : byte
    {
        /// <summary>
        /// The PCAN message is a CAN Standard Frame (11-bit identifier)
        /// </summary>
        PCAN_MESSAGE_STANDARD  = 0x00,
        /// <summary>
        /// The PCAN message is a CAN Remote-Transfer-Request Frame
        /// </summary>
        PCAN_MESSAGE_RTR       = 0x01,
        /// <summary>
        /// The PCAN message is a CAN Extended Frame (29-bit identifier)
        /// </summary>
        PCAN_MESSAGE_EXTENDED  = 0x02,
        /// <summary>
        /// The PCAN message represents a PCAN status message
        /// </summary>
        PCAN_MESSAGE_STATUS    = 0x80,
    }

    /// <summary>
    /// Represents a PCAN filter mode
    /// </summary>
    public enum TPCANMode : byte
    {
        /// <summary>
        /// Mode is Standard (11-bit identifier)
        /// </summary>
        PCAN_MODE_STANDARD = TPCANMessageType.PCAN_MESSAGE_STANDARD,
        /// <summary>
        /// Mode is Extended (29-bit identifier)
        /// </summary>
        PCAN_MODE_EXTENDED = TPCANMessageType.PCAN_MESSAGE_EXTENDED,
    }

    /// <summary>
    /// Represents a PCAN Baud rate register value
    /// </summary>
    public enum TPCANBaudrate : ushort
    {
        /// <summary>
        /// 1 MBit/s
        /// </summary>
        PCAN_BAUD_1M      = 0x0014,
        /// <summary>
        /// 800 KBit/s
        /// </summary>
        PCAN_BAUD_800K    = 0x0016,
        /// <summary>
        /// 500 kBit/s
        /// </summary>
        PCAN_BAUD_500K    = 0x001C,
        /// <summary>
        /// 250 kBit/s
        /// </summary>
        PCAN_BAUD_250K    = 0x011C,
        /// <summary>
        /// 125 kBit/s
        /// </summary>
        PCAN_BAUD_125K    = 0x031C,
        /// <summary>
        /// 100 kBit/s
        /// </summary>
        PCAN_BAUD_100K    = 0x432F,
        /// <summary>
        /// 95,238 KBit/s
        /// </summary>
        PCAN_BAUD_95K     = 0xC34E,
        /// <summary>
        /// 83,333 KBit/s
        /// </summary>
        PCAN_BAUD_83K     = 0x4B14,
        /// <summary>
        /// 50 kBit/s
        /// </summary>
        PCAN_BAUD_50K     = 0x472F,
        /// <summary>
        /// 47,619 KBit/s
        /// </summary>
        PCAN_BAUD_47K     = 0x1414,
        /// <summary>
        /// 33,333 KBit/s
        /// </summary>
        PCAN_BAUD_33K     = 0x1D14,
        /// <summary>
        /// 20 kBit/s
        /// </summary>
        PCAN_BAUD_20K     = 0x532F,
        /// <summary>
        /// 10 kBit/s
        /// </summary>
        PCAN_BAUD_10K     = 0x672F,
        /// <summary>
        /// 5 kBit/s
        /// </summary>
        PCAN_BAUD_5K      = 0x7F7F,
    }

    /// <summary>
    /// Represents the type of PCAN (non plug&play) hardware to be initialized
    /// </summary>
    public enum TPCANType : byte
    {
        /// <summary>
        /// PCAN-ISA 82C200
        /// </summary>
        PCAN_TYPE_ISA           = 0x01,
        /// <summary>
        /// PCAN-ISA SJA1000
        /// </summary>
        PCAN_TYPE_ISA_SJA       = 0x09,
        /// <summary>
        /// PHYTEC ISA 
        /// </summary>
        PCAN_TYPE_ISA_PHYTEC    = 0x04,
        /// <summary>
        /// PCAN-Dongle 82C200
        /// </summary>
        PCAN_TYPE_DNG           = 0x02,
        /// <summary>
        /// PCAN-Dongle EPP 82C200
        /// </summary>
        PCAN_TYPE_DNG_EPP       = 0x03,
        /// <summary>
        /// PCAN-Dongle SJA1000
        /// </summary>
        PCAN_TYPE_DNG_SJA       = 0x05,
        /// <summary>
        /// PCAN-Dongle EPP SJA1000
        /// </summary>
        PCAN_TYPE_DNG_SJA_EPP   = 0x06,
    }
    #endregion
	
	#region Structures
    /// <summary>
    /// Represents a PCAN message
    /// </summary>
    public struct TPCANMsg
    {
        /// <summary>
        /// 11/29-bit message identifier
        /// </summary>
        public uint ID;
        /// <summary>
        /// Type of the message
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public TPCANMessageType MSGTYPE;  
        /// <summary>
        /// Data Length Code of the message (0..8)
        /// </summary>
        public byte LEN;      
        /// <summary>
        /// Data of the message (DATA[0]..DATA[7])
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        public byte[] DATA;   
    }

    /// <summary>
    /// Represents a timestamp of a received PCAN message.
    /// Total Microseconds = micros + 1000 * millis + 0xFFFFFFFF * 1000 * millis_overflow
    /// </summary>
    public struct TPCANTimestamp
    {
        /// <summary>
        /// Base-value: milliseconds: 0.. 2^32-1
        /// </summary>
        public uint millis;             
        /// <summary>
        /// Roll-arounds of millis
        /// </summary>
        public ushort millis_overflow;  
        /// <summary>
        /// Microseconds: 0..999
        /// </summary>
        public ushort micros;
    }
    #endregion
	
	#region PCANBasic class
    /// <summary>
    /// PCAN-Basic API class implementation
    /// </summary>
    public static class PCANBasic
    {
        #region PCAN-BUS Handles Definition
        /// <summary>
        /// Undefined/default value for a PCAN bus
        /// </summary>
        public const TPCANHandle PCAN_NONEBUS = 0x00;
        
        /// <summary>
        /// PCAN-ISA interface, channel 1
        /// </summary>
        public const TPCANHandle PCAN_ISABUS1 = 0x21;
        /// <summary>
        /// PCAN-ISA interface, channel 2
        /// </summary>
        public const TPCANHandle PCAN_ISABUS2 = 0x22;
        /// <summary>
        /// PCAN-ISA interface, channel 3
        /// </summary>
        public const TPCANHandle PCAN_ISABUS3 = 0x23;
        /// <summary>
        /// PCAN-ISA interface, channel 4
        /// </summary>
        public const TPCANHandle PCAN_ISABUS4 = 0x24;
        /// <summary>
        /// PCAN-ISA interface, channel 5
        /// </summary>
        public const TPCANHandle PCAN_ISABUS5 = 0x25;
        /// <summary>
        /// PCAN-ISA interface, channel 6
        /// </summary>
        public const TPCANHandle PCAN_ISABUS6 = 0x26;
        /// <summary>
        /// PCAN-ISA interface, channel 7
        /// </summary>
        public const TPCANHandle PCAN_ISABUS7 = 0x27;
        /// <summary>
        /// PCAN-ISA interface, channel 8
        /// </summary>
        public const TPCANHandle PCAN_ISABUS8 = 0x28;
        
        /// <summary>
        /// PPCAN-Dongle/LPT interface, channel 1 
        /// </summary>
        public const TPCANHandle PCAN_DNGBUS1 = 0x31;

        /// <summary>
        /// PCAN-PCI interface, channel 1
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS1 = 0x41;
        /// <summary>
        /// PCAN-PCI interface, channel 2
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS2 = 0x42;
        /// <summary>
        /// PCAN-PCI interface, channel 3
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS3 = 0x43;
        /// <summary>
        /// PCAN-PCI interface, channel 4
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS4 = 0x44;
        /// <summary>
        /// PCAN-PCI interface, channel 5
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS5 = 0x45;
        /// <summary>
        /// PCAN-PCI interface, channel 6
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS6 = 0x46;
        /// <summary>
        /// PCAN-PCI interface, channel 7
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS7 = 0x47;
        /// <summary>
        /// PCAN-PCI interface, channel 8
        /// </summary>
        public const TPCANHandle PCAN_PCIBUS8 = 0x48;

        /// <summary>
        /// PCAN-USB interface, channel 1
        /// </summary>
        public const TPCANHandle PCAN_USBBUS1 = 0x51;
        /// <summary>
        /// PCAN-USB interface, channel 2
        /// </summary>
        public const TPCANHandle PCAN_USBBUS2 = 0x52;
        /// <summary>
        /// PCAN-USB interface, channel 3
        /// </summary>
        public const TPCANHandle PCAN_USBBUS3 = 0x53;
        /// <summary>
        /// PCAN-USB interface, channel 4
        /// </summary>
        public const TPCANHandle PCAN_USBBUS4 = 0x54;
        /// <summary>
        /// PCAN-USB interface, channel 5
        /// </summary>
        public const TPCANHandle PCAN_USBBUS5 = 0x55;
        /// <summary>
        /// PCAN-USB interface, channel 6
        /// </summary>
        public const TPCANHandle PCAN_USBBUS6 = 0x56;
        /// <summary>
        /// PCAN-USB interface, channel 7
        /// </summary>
        public const TPCANHandle PCAN_USBBUS7 = 0x57;
        /// <summary>
        /// PCAN-USB interface, channel 8
        /// </summary>
        public const TPCANHandle PCAN_USBBUS8 = 0x58;

        /// <summary>
        /// PCAN-PC Card interface, channel 1
        /// </summary>
        public const TPCANHandle PCAN_PCCBUS1 = 0x61;
        /// <summary>
        /// PCAN-PC Card interface, channel 2
        /// </summary>
        public const TPCANHandle PCAN_PCCBUS2 = 0x62;
        #endregion
    	
    	#region Parameter values definition
        /// <summary>
        /// The PCAN parameter is not set (inactive)
        /// </summary>
        public const int PCAN_PARAMETER_OFF = 0;
        /// <summary>
        /// The PCAN parameter is set (active)
        /// </summary>
        public const int PCAN_PARAMETER_ON = 1;
        /// <summary>
        /// The PCAN filter is closed. No messages will be received
        /// </summary>
        public const int PCAN_FILTER_CLOSE = 0;
        /// <summary>
        /// The PCAN filter is fully opened. All messages will be received
        /// </summary>
        public const int PCAN_FILTER_OPEN = 1;
        /// <summary>
        /// The PCAN filter is custom configured. Only registered 
        /// messages will be received
        /// </summary>
        public const int PCAN_FILTER_CUSTOM = 2;
        /// <summary>
        /// The PCAN-Channel handle is illegal, or its associated hadware is not available
        /// </summary>
        public const int PCAN_CHANNEL_UNAVAILABLE = 0;
        /// <summary>
        /// The PCAN-Channel handle is available to be connected (Plug&Play Hardware: it means futhermore that the hardware is plugged-in)
        /// </summary>
        public const int PCAN_CHANNEL_AVAILABLE = 1;
        /// <summary>
        /// The PCAN-Channel handle is valid, and is already being used
        /// </summary>
        public const int PCAN_CHANNEL_OCCUPIED = 2;

        /// <summary>
        /// Logs system exceptions / errors
        /// </summary>
        public const int LOG_FUNCTION_DEFAULT = 0x00;
        /// <summary>
        /// Logs the entries to the PCAN-Basic API functions 
        /// </summary>
        public const int LOG_FUNCTION_ENTRY = 0x01;
        /// <summary>
        /// Logs the parameters passed to the PCAN-Basic API functions 
        /// </summary>
        public const int LOG_FUNCTION_PARAMETERS = 0x02;
        /// <summary>
        /// Logs the exits from the PCAN-Basic API functions 
        /// </summary>
        public const int LOG_FUNCTION_LEAVE = 0x04;
        /// <summary>
        /// Logs the CAN messages passed to the CAN_Write function
        /// </summary>
        public const int LOG_FUNCTION_WRITE = 0x08;
        /// <summary>
        /// Logs the CAN messages received within the CAN_Read function
        /// </summary>
        public const int LOG_FUNCTION_READ = 0x10;
        /// <summary>
        /// Logs all possible information within the PCAN-Basic API functions
        /// </summary>
        public const int LOG_FUNCTION_ALL = 0xFFFF;
        #endregion
    	
    	#region PCANBasic API Implementation
        /// <summary>
        /// Initializes a PCAN Channel 
        /// </summary>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="Btr0Btr1">The speed for the communication (BTR0BTR1 code)</param>
        /// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="Interrupt">NON PLUG&PLAY: Interrupt number of the parallel por</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_Initialize")]
        public static extern TPCANStatus Initialize(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            [MarshalAs(UnmanagedType.U2)]
            TPCANBaudrate Btr0Btr1,
            [MarshalAs(UnmanagedType.U1)]
            TPCANType HwType, 
            UInt32 IOPort, 
            UInt16 Interrupt);

        /// <summary>
        /// Initializes a PCAN Channel
        /// </summary>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="Btr0Btr1">The speed for the communication (BTR0BTR1 code)</param>
        /// <returns>A TPCANStatus error code</returns>
        public static TPCANStatus Initialize(
            TPCANHandle Channel,
            TPCANBaudrate Btr0Btr1)
        {
            return Initialize(Channel, Btr0Btr1, (TPCANType)0, 0, 0);
        }

        /// <summary>
        /// Uninitializes one or all PCAN Channels initialized by CAN_Initialize
        /// </summary>
        /// <remarks>Giving the TPCANHandle value "PCAN_NONEBUS", 
        /// uninitialize all initialized channels</remarks>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_Uninitialize")]
        public static extern TPCANStatus Uninitialize(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel);

        /// <summary>
        /// Resets the receive and transmit queues of the PCAN Channel
        /// </summary>
        /// <remarks>A reset of the CAN controller is not performed</remarks>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_Reset")]
        public static extern TPCANStatus Reset(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel);

        /// <summary>
        /// Gets the current status of a PCAN Channel
        /// </summary>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_GetStatus")]
        public static extern TPCANStatus GetStatus(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel);

        /// <summary>
        /// Reads a CAN message from the receive queue of a PCAN Channel
        /// </summary>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="MessageBuffer">A TPCANMsg structure buffer to store the CAN message</param>
        /// <param name="TimestampBuffer">A TPCANTimestamp structure buffer to get
        /// the reception time of the message</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_Read")]
        public static extern TPCANStatus Read(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            out TPCANMsg MessageBuffer,
            out TPCANTimestamp TimestampBuffer);

        [DllImport("PCANBasic.dll", EntryPoint = "CAN_Read")]
        private static extern TPCANStatus Read(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            out TPCANMsg MessageBuffer,
            IntPtr bufferPointer);

        /// <summary>
        /// Reads a CAN message from the receive queue of a PCAN Channel
        /// </summary>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="MessageBuffer">A TPCANMsg structure buffer to store the CAN message</param>        
        /// <returns>A TPCANStatus error code</returns>
        public static TPCANStatus Read(
            TPCANHandle Channel,
            out TPCANMsg MessageBuffer)
        {
            return Read(Channel, out MessageBuffer, IntPtr.Zero);
        }

        /// <summary>
        ///  Transmits a CAN message 
        /// </summary>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="MessageBuffer">A TPCANMsg buffer with the message to be sent</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_Write")]
        public static extern TPCANStatus Write(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            ref TPCANMsg MessageBuffer);

        /// <summary>
        /// Configures the reception filter
        /// </summary>
        /// <remarks>The message filter will be expanded with every call to 
        /// this function. If it is desired to reset the filter, please use
        /// the 'SetValue' function</remarks>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="FromID">The lowest CAN ID to be received</param>
        /// <param name="ToID">The highest CAN ID to be received</param>
        /// <param name="Mode">Message type, Standard (11-bit identifier) or
        /// Extended (29-bit identifier)</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_FilterMessages")]
        public static extern TPCANStatus FilterMessages(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            UInt32 FromID,
            UInt32 ToID,
            [MarshalAs(UnmanagedType.U1)]
            TPCANMode Mode);

        /// <summary>
        /// Retrieves a PCAN Channel value
        /// </summary>
        /// <remarks>Parameters can be present or not according with the kind 
        /// of Hardware (PCAN Channel) being used. If a parameter is not available,
        /// a PCAN_ERROR_ILLPARAMTYPE error will be returned</remarks>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="Parameter">The TPCANParameter parameter to get</param>
        /// <param name="StringBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_GetValue")]
        public static extern TPCANStatus GetValue(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            [MarshalAs(UnmanagedType.U1)]
            TPCANParameter Parameter,
            StringBuilder StringBuffer,            
            UInt32 BufferLength);

        /// <summary>
        /// Retrieves a PCAN Channel value
        /// </summary>
        /// <remarks>Parameters can be present or not according with the kind 
        /// of Hardware (PCAN Channel) being used. If a parameter is not available,
        /// a PCAN_ERROR_ILLPARAMTYPE error will be returned</remarks>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="Parameter">The TPCANParameter parameter to get</param>
        /// <param name="NumericBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_GetValue")]
        public static extern TPCANStatus GetValue(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            [MarshalAs(UnmanagedType.U1)]
            TPCANParameter Parameter,
            out UInt32 NumericBuffer,
            UInt32 BufferLength);

        /// <summary>
        /// Configures or sets a PCAN Channel value 
        /// </summary>
        /// <remarks>Parameters can be present or not according with the kind 
        /// of Hardware (PCAN Channel) being used. If a parameter is not available,
        /// a PCAN_ERROR_ILLPARAMTYPE error will be returned</remarks>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="Parameter">The TPCANParameter parameter to set</param>
        /// <param name="NumericBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_SetValue")]
        public static extern TPCANStatus SetValue(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            [MarshalAs(UnmanagedType.U1)]
            TPCANParameter Parameter,
            ref UInt32 NumericBuffer,
            UInt32 BufferLength);

        /// <summary>
        /// Configures or sets a PCAN Channel value
        /// </summary>
        /// <remarks>Parameters can be present or not according with the kind 
        /// of Hardware (PCAN Channel) being used. If a parameter is not available,
        /// a PCAN_ERROR_ILLPARAMTYPE error will be returned</remarks>
        /// <param name="Channel">The handle of a PCAN Channel</param>
        /// <param name="Parameter"></param>
        /// <param name="StringBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_SetValue")]
        public static extern TPCANStatus SetValue(
            [MarshalAs(UnmanagedType.U1)]
            TPCANHandle Channel,
            [MarshalAs(UnmanagedType.U1)]
            TPCANParameter Parameter,
            [MarshalAs(UnmanagedType.LPStr,SizeParamIndex=3)]
            string StringBuffer,
            UInt32 BufferLength);

        /// <summary>
        /// Returns a descriptive text of a given TPCANStatus error 
        /// code, in any desired language
        /// </summary>
        /// <remarks>The current languages available for translation are: 
        /// Neutral (0x00), German (0x07), English (0x09), Spanish (0x0A),
        /// Italian (0x10) and French (0x0C)</remarks>
        /// <param name="Error">A TPCANStatus error code</param>
        /// <param name="Language">Indicates a 'Primary language ID'</param>
        /// <param name="StringBuffer">Buffer for the text (must be at least 256 in length)</param>
        /// <returns>A TPCANStatus error code</returns>
        [DllImport("PCANBasic.dll", EntryPoint = "CAN_GetErrorText")]
        public static extern TPCANStatus GetErrorText(
            [MarshalAs(UnmanagedType.U4)]
            TPCANStatus Error,
            UInt16 Language,
            StringBuilder StringBuffer);
        #endregion
    }
    #endregion
}