//@author: Weyland Yutani
//@help: template for standard shaders
//@tags: template
//@credits:


Texture2D texture2d <string uiname="Texture";>;

SamplerState linearSampler : IMMUTABLE
{
	Filter = MIN_MAG_MIP_LINEAR;
	AddressU = Clamp;
	AddressV = Clamp;
};

cbuffer cbPerDraw : register( b0 )
{
	float4x4 tVP : VIEWPROJECTION;
};

cbuffer cbPerObj : register( b1 )
{
	float4x4 tW : WORLD;
};

struct VS_IN
{
	float4 PosO : POSITION;
	float4 TexCd : TEXCOORD0;
	
};

struct vs2ps
{
	float4 PosWVP: SV_POSITION;
	float4 TexCd: TEXCOORD0;
};

vs2ps VS(VS_IN input)
{
	vs2ps output;
	output.PosWVP  = mul(input.PosO,mul(tW,tVP));
	output.TexCd = input.TexCd;
	return output;
}

// -------------------------------- Vars ---------------------------------------

float time;

static const int max_iterations = 45;
static const float stop_threshold = 0.001;
static const float grad_step = 0.001;
static const float clip_far = 4.0;
static const float PI = 3.14159265359;

static const float furDepth = 0.5;
static const int furLayers = 50;
static const float rayStep = furDepth * 2.0 / float(furLayers);
static const float furThreshold = 0.4;
static const float uvScale = 0.5;

float r = 1.0;
float rotation = 60.0;// * (time);				// add Time in the methods!!!
float elevation = 0.25;// * cos(time * 2.5);

// ------------------------------- Methods -------------------------------------

float3x3 rotX(float g) {
	g = radians(g);
	float2 a = float2(cos(g), sin(g));
	return float3x3(1.0, 0.0, 0.0,
					0.0, a.x, -a.y,
					0.0, a.y, a.x);
}

float3x3 rotY(float g) {
	g = radians(g);
	float2 a = float2(cos(g), sin(g));
	return float3x3(a.x, 0.0, a.y,
	0.0, 1.0, 0.0,
	-a.y, 0.0, a.x);
}

float3 map(float3 p) {
	p.y -= elevation;
//	p *= rotY(rotation);
	p = mul(p, rotY(rotation));
	float s = (length(p) - r) - 0.12 * sin(6. * p.x) * cos(6. * p.y)* sin(6. * p.z);
	return float3(s * 0.5, s * 0.5, s * 0.5);
}


float2 cartesianToSpherical(float3 p) {
	float r = length(p);
	p = normalize(p);
	float t = (r - (1.0 - furDepth)) / furDepth;
//	p.zyx *= rotX(rotation);
	p.zxy = mul(p.zxy, rotX(rotation) );
//	float2 uv = float2(atan(p.y, p.x), acos(p.z));
	float2 uv = float2(atan2(p.y, p.x), acos(p.z));
	
	uv.y -= t * t * elevation;	
	return uv;
}

float furDensity(float3 pos) {
	float2 uv = cartesianToSpherical(pos.xzy);
//	float4 tex = texture2D(iChannel0, uv * uvScale);
	float4 tex = texture2d.Sample(linearSampler, uv);
	
	float density = smoothstep(furThreshold, 1.0, tex.x);
	
	pos.y -= elevation;
	float3 p = pos;
//	p *= rotY(rotation);
	p = mul(p, rotY(rotation) );
	float r = length(pos) - 0.12 * sin(6.0 * p.x) * cos(6.0 * p.y)* sin(6.0 * p.z);;
	float t = (r - (1.0 - furDepth)) / furDepth;
	
	return density * (1.0 - t);
	
}

float3 furNormal(float3 pos, float density) {
	float eps = 0.0001;
	float3 n;
	n.x = furDensity( float3(pos.x + eps, pos.y, pos.z)) - density;
	n.y = furDensity( float3(pos.x, pos.y + eps, pos.z)) - density;
	n.z = furDensity( float3(pos.x, pos.y, pos.z + eps)) - density;
	return normalize(n);
}


float3 furShade(float3 pos, float3 color, float3 light, float3 eye, float density) {
	
	float3 v = normalize(light - pos);
	float3 n = furNormal(pos, density);
	float3 ev = normalize(pos - eye);
	float3 h = reflect(ev, n);
	
	float diff = max(0.0, dot(v, n)) + 0.4;
	float spec = pow(max(0.0, dot(v, h)), 64.);
	
	float r = length(pos);
	float t = (r - (1.0 - furDepth)) / furDepth;
	t = clamp(t, 0.3, 1.);
	
//	diff = mix(diff, 1., 0.5);
	diff = lerp(diff, 1., 0.5);
	
	return color * t * (diff + 1.9 * spec);
}

float3 gradient( float3 v ) {
	const float3 delta = float3( grad_step, 0.0, 0.0 );
	float va = map( v ).x;
	return normalize (
	float3(
	map( v + delta.xyy).x - va,
	map( v + delta.yxy).x - va,
	map( v + delta.yyx).x - va
	)
	);
}

float3 ray_marching( float3 origin, float3 dir, float start, float end ) {
	
	float depth = start;
//	float3 salida = float3(end);
	float3 salida = float3(end, end, end);
	float3 dist = float3(0.1, 0.1, 0.1);
	
	for ( int i = 0; i < max_iterations; i++ ) 		{
		if ( dist.x < stop_threshold || depth > end ) break;
		dist = map( origin + dir * depth );
		depth += dist.x;
	}
	
	salida = float3(depth, dist.y, dist.z);
	return salida;
}

float3 shading( float3 v, float3 n, float3 eye, float3 lightMix) {
	
	float3 final = float3( 0.0, 0.0, 0.0 );
	float3 ev = normalize( v - eye );
	float3 ref_ev = reflect( ev, n );
	float3 light_pos   = float3(0.0, 2.0, -2.0);
	float3 vl = normalize( light_pos - v );
	float diffuse  = max( 0.0, dot( vl, n ) );
	float specular = max( 0.0, dot( vl, ref_ev ) );
	specular = pow( specular, lightMix.x );
	final += float3( 0.9, 0.9, 0.9 ) * ( diffuse * lightMix.y + specular * lightMix.z);
	final += float3(0.2, 0.2, 0.2);
	
	return final;
}

float3 ray_dir( float fov, float2 size, float2 pos ) {
	float2 xy = pos - size * 0.5;
	
	float cot_half_fov = tan(radians( 90.0 - fov * 0.5 ));
	float z = size.y * 0.5 * cot_half_fov;
	
	return normalize( float3( xy, z ) );
}





// -------------------------------- PS -----------------------------------------
float4 PSFur(vs2ps In): SV_Target
{
	
//	float3 rd = ray_dir(60.0, iResolution.xy, gl_FragCoord.xy );
		
	float2 p = In.TexCd.rg;
	float3 rd = ray_dir(60.0, p.r, p.g );
		float3 eye = float3( 0.0, 0.0, -2.8 );
			
		float3 lightMix = float3(6.0, 0.45, 0.15);
		float3 color = float3(0.0, 0.0, 0.0);

		float3 data = ray_marching( eye, rd, 0.0, clip_far );
		if ( data.x < clip_far ) {
			
			float3 pos = eye + rd * data.x;
			float3 n = gradient( pos );
			
			float3 lightColor =  shading( pos, n, eye, lightMix) * 2.0;			
			color = float3(1.0, 1.0, 1.0) * lightColor;
				
			float4 c = float4(0.0, 0.0, 0.0, 0.0);
			
			for(int i=0; i<furLayers; i++) {
				float4 sampleCol;
				sampleCol.a = furDensity(pos);
				if (sampleCol.a > 0.) {
					sampleCol.rgb = sampleCol.a * furShade(pos, float3(85., 72., 56) * lightColor / 255., float3(0.0, 2.0, -2.0), eye, sampleCol.a);
					c = c + sampleCol * (1.0 - c.a);
				}
				pos += rd * rayStep;
			}
			
			color = c.xyz;
		}
			
		return float4(2. * color, 1.0 );
}


technique10 Voronoi
{
	pass P0
	{
		SetVertexShader( CompileShader( vs_5_0, VS() ) );
		SetPixelShader( CompileShader( ps_5_0, PSFur() ) );
	}
}






