//@name: Infinite City
//@author: Psylion
//@help: template for standard shaders
//@tags: template
//@credits: Shadertoy - Infinite City - Ben Weston - 2013 
//@License: License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.


Texture2D texture2d <string uiname="Texture";>;

SamplerState linearSampler : IMMUTABLE
{
	Filter = MIN_MAG_MIP_LINEAR;
	//	Filter = Anisotropic;
	AddressU = Wrap;
	AddressV = Wrap;
};
float2 R:TARGETSIZE;

cbuffer cbPerDraw : register( b0 ){
	float4x4 tVP : VIEWPROJECTION;	
	float4x4 tW:WORLD;
	float4x4 tV:VIEW;
	float4x4 tP:PROJECTION;
	float4x4 tWI:WORLDINVERSE;
	float4x4 tVI:VIEWINVERSE;
	float4x4 tPI:PROJECTIONINVERSE;
};

struct VS_IN{
	float4 PosO : POSITION;
	float4 TexCd : TEXCOORD0;

};

struct vs2ps{
    float4 PosWVP: SV_POSITION;
    float2 TexCd: TEXCOORD0;
};

vs2ps VS(VS_IN input){
    vs2ps output;
    output.PosWVP  = mul(input.PosO,tW);
	
    output.TexCd = input.TexCd.xy;
    return output;
}

float3 UVtoEYE(float2 UV){
	return normalize(mul(float4(mul(float4((UV.xy*2-1)*float2(1,-1),0,1),tPI).xy,1,0),tVI).xyz);
}

float time;

float2 Rand( float2 pos){
	//float2 c0=texture2d.SampleLevel(linearSampler,In.TexCd, 0);
	//float2 cord = In.TexCd;
	//c0 = (pos+.5/64.0,-100.0);
	//float4 r = (linearSampler, c0, 0);
	//return r;
	return texture2d.SampleLevel(linearSampler, (pos+.5)/256.0, -100.0 ).xy;
}

// return float3
float3 VoronoiPoint(float2 pos, float2 delta, vs2ps In )
{
	const float randScale = .8; // reduce this to remove axis-aligned hard edged errors
	
	float2 p = floor(pos)+delta;
	float2 r = (Rand(p)-.5)*randScale;
	float2 c = p+.5+r;
	
	// various length calculations for different patterns
	//float l = length(c-pos);
	//float l = length(vec3(c-pos,.1));
	float l = abs(c.x-pos.x)+abs(c.y-pos.y); // more interesting shapes
	
	return float3(c,l);
}

// For building height I want to know which voronoi point I used
// For side-walls I want difference between distance of closest 2 points
float3 Voronoi( float2 pos, vs2ps In )
{
	// find closest & second closest points
	float3 delta = float3(-1,0,1);

	// sample surrounding points on the distorted grid
	// could get 2 samples for the price of one using a rotated (17,37) grid...
	float3 p[9];
	p[0] = VoronoiPoint( pos, delta.xx, In);
	p[1] = VoronoiPoint( pos, delta.yx, In );
	p[2] = VoronoiPoint( pos, delta.zx, In );
	p[3] = VoronoiPoint( pos, delta.xy, In );
	p[4] = VoronoiPoint( pos, delta.yy, In );
	p[5] = VoronoiPoint( pos, delta.zy, In );
	p[6] = VoronoiPoint( pos, delta.xz, In );
	p[7] = VoronoiPoint( pos, delta.yz, In );
	p[8] = VoronoiPoint( pos, delta.zz, In );

	float3 closest;
	closest.z =
		min(
			min(
				min(
					min( p[0].z, p[1].z ),
					min( p[2].z, p[3].z )
				), min(
					min( p[4].z, p[5].z ),
					min( p[6].z, p[7].z )
				)
			), p[8].z
		);
	
	// find second closest
	// maybe there's a better way to do this
	closest.xy = p[8].xy;
	for ( int i=0; i < 8; i++ )
	{
		if ( closest.z == p[i].z )
		{
			closest = p[i];
			p[i] = p[8];
		}
	}
		
	float t;
	t = min(
			min(
				min( p[0].z, p[1].z ),
				min( p[2].z, p[3].z )
			), min(
				min( p[4].z, p[5].z ),
				min( p[6].z, p[7].z )
			)
		);

	/*slower:
	float t2 = 9.0;
	vec3 closest = point[8];
	for ( int i=0; i < 8; i++ )
	{
		if ( point[i].z < closest.z )
		{
			t2 = closest.z;
			closest = point[i];
		}
		else if ( point[i].z < t2 )
		{
			t2 = point[i].z;
		}
	}*/
	
	return float3( closest.xy, t-closest.z );
}
/*
float mix ( float x, float y, float2 a ){
	return (1-a)*x + a*y;
}
*/
#define mix(a,b,c) lerp(a,b,c)
float DistanceField( float3 pos, float dist, vs2ps In )
{
	float3 v = Voronoi(pos.xz, In);
	float2 r = Rand(v.xy*4.0); // per-building seed
	
	float f = (.2+.3*r.y-v.z)*.5; //.7071; // correct for max gradient of voronoi x+z distance calc
	
	// random height
	float h = r.x; // v.xy is position of cell centre, use it as random seed
	//h = mix(.2,2.0,pow(h,2.0)); // GLSL -> HLSL ->T mix ( T x, T y, T2 a ){return (1-a)*x + a*y;}
	//h = (1-pow(h,2.0)*.2+pow(h,2.0)*.2);
	h = mix(.2,2.0,pow(h,2.0));
	h = pos.y-h;

	// we get precision problems caused by the discontinuity in height
	// so clamp it near to the surface and then apply a plane at max height	
	h = max( min( h, .008*dist ), pos.y-2.0 );

//	f = max( f, h );
	if ( f > 0.0 && h > 0.0 )
		f = sqrt(f*f+h*h); // better distance computation, to reduce errors
	else
		f = max(f,h);
	
	f = min( f, pos.y ); // ground plane
	
	return f;
}

float DistanceField( float3 pos, vs2ps In ){
	return DistanceField( pos, 10.0, In );
}

// normal
// could do this analytically, by looking at the maths when comupting voronoi value
// but the lions share of the cost is in the trace, not this, so I shalln't worry
float3 GetNormal( float3 pos, vs2ps In ){
	float3 n;
	float2 delta = float2(0,1);

	// allow a slight curve so it looks more interesting
	#ifdef FIRST_PERSON
		delta *= .004;
	#else
		delta *= .04;
	#endif
	
	n.x = DistanceField( pos+delta.yxx, In ) - DistanceField( pos-delta.yxx, In );
	n.y = DistanceField( pos+delta.xyx, In ) - DistanceField( pos-delta.xyx, In );
	n.z = DistanceField( pos+delta.xxy, In ) - DistanceField( pos-delta.xxy, In );

	// we get some black on surfaces because of flat spots in the voronoi
	// fix that by giving it an arbitrary (and incorrect) normal
	if ( dot(n,n) == 0.0 )
		n.y += 1.0;

	return normalize(n);
}
float3 iResolution:TARGETSIZE;

float4 PSInfiniteCity(vs2ps In): SV_Target{
	
	// float2 mouse = iMouse.xy/iResolution.xy;
	
	float h;
	/*
	#ifdef FIRST_PERSON
		float2 rot = float2(-.2,.0)+float2(6.28,-1.5); //*mouse
		
	//	vec3 rayStart = vec3(.1,.03,.1);//vec3(0,10,0) + vec3(1,0,1)*iGlobalTime + vec3(0,-8,0)*mouse.y;
	//	rayStart += -10.0*vec3(sin(rot.x),0,cos(rot.x));
	
		float3 rayStart = float3(0,.03,0) + time*float3(-.02,0,.02);
		// find closest road to the right
		h = 1.0;
		for ( int i=0; i < 40; i++ )
		{
			if ( h < .01 )
				break;
			h = Voronoi( rayStart.xz ).z*.3;
			rayStart.xz += h;
		}
	
		float zoom = .7;
	#else
		float2 rot = float2(-.2,.28)+float2(1.6,.5); //*mouse
		
		float3 rayStart = float3(0,5,0) + float3(1,0,1)*time + float3(0,6,0);//*mouse.y
		rayStart += -10.0*float3(sin(rot.x),0,cos(rot.x));

		float zoom = 1.0;
	#endif
	*/
	
	//float3 rayDir = normalize( float3( In.PosWVP.xy-iResolution.xy*.5, iResolution.x * zoom ) );
	//float3 rayDir = normalize( float3( iResolution.xy-iResolution.xy*.5, iResolution.x * zoom ) );
	//rayDir.yz = rayDir.yz*cos(rot.y)+rayDir.zy*sin(rot.y)*float2(-1,1);
	//rayDir.xz = rayDir.xz*cos(rot.x)+rayDir.zx*sin(rot.x)*float2(1,-1);
	
	float3 rayDir=UVtoEYE(In.PosWVP.xy/iResolution.xy);
	float3 rayStart=tVI[3].xyz;
	
	// trace
	float t = 0.0;
	h = 1.0;
	for ( int i=0; i < 100; i++ )
	{
		if ( h < .003 )
			break;
		
		h = DistanceField(rayStart+rayDir*t, t, In);
		t += h;
	}
	
	// shade
	float3 pos = rayStart+rayDir*t;
	float3 norm = GetNormal(pos, In);
	float3 v = Voronoi(pos.xz, In);
	
	float2 r = Rand( v.xy).xy;
	float4 albedo = mix( mix( float4(.4,.2,.0,0), float4(1,1,1,0), r.x),
					   mix( float4(0,.2,.6,1), float4(0,0,0,1), r.x),
					   r.y );
	
	// floors
	if ( frac(pos.y*8.0) < .4 )
		albedo = mix( float4(0,0,0,0), float4(1,1,1,0), r.x );

	// remove reflection on rooves!
	albedo.w = mix ( albedo.w, 0.0, step( .2, norm.y ) );
	
	// roads
	albedo = mix( float4(.05,.05,.05,0), albedo, step( .07, abs(v.z-.08) ) );
	
	float3 lighting = max(0.0,dot(norm,normalize(float3(-2,3,-1))))*float3(1,.9,.8);
	lighting += float3(.2,0,0); //ambient
	
	float3 result = albedo.xyz * lighting;
	
	// reflections
	float fresnel = pow(1.0+dot(norm,rayDir),5.0);
	if ( fresnel > 1.0 ) fresnel = 0.0;
	fresnel = mix( .2, 1.0, fresnel );
	
	float3 reflection = texture2d.SampleLevel(linearSampler, reflect(rayDir,norm).xz/8.0,0 ).rgb*3.0;
	result = mix( result, reflection, fresnel*albedo.w );

	if ( h > .01 )//&& rayDir.y > 0.0 )
	{
		// sky
		result = mix( float3(.3,0,0), float3(.85,1.0,1.2), smoothstep( -.1,.1,rayDir.y ) );
	}
	
	// fake ambient occlusion
	result *= mix( .2, 1.0, smoothstep( 0.0, .7, pos.y ) );
	
	// fog
	//result = mix( vec3(.85,1.0,1.2), result, exp(-t*.02) );
	//More realistic fog
	result *= pow( float3(.7,.57,.3), float3(t*.02,0,0) ); // absorb more blue on distant things
	result += float3(.7,.9,1.2)*(1.0-exp(-t*.02));
	//result=reflection;
	//result=normalize(rayDir);
	return float4(result,1);
}

technique10 InfiniteCity
{
	pass P0
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );
		SetPixelShader( CompileShader( ps_4_0, PSInfiniteCity()));
	}
}


