// Ben Quantock 2015
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
// https://creativecommons.org/licenses/by-nc-sa/3.0/

float2 R:TARGETSIZE;
Texture2D tex0 <string uiname="Texture1";>;
Texture2D tex1 <string uiname="Texture2";>;
SamplerState s0  <bool visible=false;string uiname="Sampler";> 
{
	Filter=MIN_MAG_MIP_LINEAR;
	AddressU=wrap;
	AddressV=wrap;
};

cbuffer cbPerDraw:register( b0 )
{
float4x4 tVP:VIEWPROJECTION;	
float4x4 tW:WORLD;
float time;
float3 mouse;
float3 lightDir; 
float specPower = 400.0;
};

static float tau = 6.2831853;
static float phi = 1.61803398875;

float2 Noise( in float3 x )
{
    float3 p = floor(x);
    float3 f = frac(x);
	f = f*f*(3.0-2.0*f);
	float2 uv = (p.xy+float2(37.0,17.0)*p.z) + f.xy;
	float4 rg = tex0.Sample( s0, (uv+0.5)/256.0, -100.0 );
	return lerp( rg.yw, rg.xz, f.z );
}

float4 Map( float3 p3 )
{
    //construct an orthonormal frame for the slice
	//um... there's no 4D cross product, is there? There can't be! Would need 3 inputs. balls.
    float4 i = float4(1,0,0,0);
    float4 j = float4(0,1,0,0);
    float4 k = float4(0,0,1,0);
    
    // rules for rotation should still behave afaik
    float a,c;
    float2 s;
    a = time*.1;
    c = cos(a); s = float2(1,-1)*sin(a);
    i.xz = i.xz*c + i.zx*s;
    j.xz = j.xz*c + j.zx*s;
    k.xz = k.xz*c + k.zx*s;
    i.yw = i.yw*c + i.wy*s;
    j.yw = j.yw*c + j.wy*s;
    k.yw = k.yw*c + k.wy*s;
    i.xw = i.xw*c + i.wx*s;
    j.xw = j.xw*c + j.wx*s;
    k.xw = k.xw*c + k.wx*s;
    i.yz = i.yz*c + i.zy*s;
    j.yz = j.yz*c + j.zy*s;
    k.yz = k.yz*c + k.zy*s;
    
    // form a basis perp to float4(1,1,1,1)
	// actually easier than it sounds, just make it so dot prods sum to 0
/*    float4 i = float4(1,1,-1,-1)/2.0;
    float4 j = float4(-1,1,-1,1)/2.0;
    float4 k = float4(1,-1,-1,1)/2.0;*/
    
    //float4 p = float4(p3,0);
    float4 p = i*p3.x + j*p3.y + k*p3.z;// + float4(1,1,1,1)*.5*.25;

    return p;
}
    
float DistanceField( float3 p3 )
{
    float4 p = Map(p3);
    
    // offset to centre of cell
    float f = 100.0;
    
    // cells
    float4 o = .5 - abs(frac(p+.5)-.5);
    f = min(f, .2 - min(min(min(o.x, o.y), o.z), o.w ) );
        
	// or grid edges (is this meaningful? - apparently, yes)
    f = min(f, min(min(min(min(min(length(o.xy),length(o.xz)),length(o.xw)),length(o.yz)),length(o.yw)),length(o.zw)) - .03);
    
    p = abs(p);
    return max( f, max(max(max(p.x,p.y),p.z),p.w)-1.53 );
}

float3 Sky( float3 ray )
{
	return lerp( float(.8), float(0), exp2(-(1.0/max(-ray.y,.01))*float3(.4,.6,1.0)));
}


float3 Shade( float3 pos, float3 ray, float3 normal, float3 lightCol, float shadowMask, float distance )
{
    float4 p = Map(pos);

    float4 c = (p+1.5)/3.0;
    //float3 albedo = (c.xyz);// + c.www)*.5;//float3(.25);
    //float3 albedo = float3(.75,0,0)*c.x + float3(.25,.5,0)*c.y + float3(0,.5,.25)*c.z + float3(0,0,.75)*c.w;
    
    float4 o = .5 - abs(frac(p+.5)-.5);
    float3 albedo = lerp( float3(1,0,0), float(1), step(.1,min(min(min(o.x, o.y), o.z), o.w )) );


	// direct light
	float ndotl = max(.0,dot(normal,lightDir));
	float lightCut = smoothstep(.0,.1,ndotl);//pow(ndotl,2.0);
	float3 light = lightCol*shadowMask*ndotl;


	// ambient light
	float3 ambient = lerp(  float3(.2,.27,.4), float(.4), (-normal.y*.5+.5) ); // ambient

	// ambient occlusion, based on my DF Lighting: https://www.shadertoy.com/view/XdBGW3
	float aoRange = distance/20.0;
	float occlusion = max( 0.0, 1.0 - DistanceField( pos + normal*aoRange )/aoRange ); // can be > 1.0
	occlusion = exp2( -2.0*pow(occlusion,2.0) ); // tweak the curve
	ambient *= occlusion;


	// subsurface scattering
	float transmissionRange = 0.1;
	float transmission = DistanceField( pos + lightDir*transmissionRange )/transmissionRange;
	float3 sslight = lightCol * smoothstep(0.0,1.0,transmission);
	float3 subsurface = float3(1,.8,.5) * sslight;


	// specular
	float3 h = normalize(lightDir-ray);
    float specFres = lerp( .02, 1.0, pow( 1.0 - dot(h,lightDir), 5.0 ) );
	float3 specular = lightCol*shadowMask*pow(max(.0,dot(normal,h))*lightCut, specPower)*specFres*(specPower+6.0)/32.0;
	

	// reflections
	float3 rray = reflect(ray,normal);
	float3 reflection = Sky( rray );
	
	// reflection occlusion, adjust the divisor for the gradient we expect
	float specOcclusion = max( 0.0, 1.0 - DistanceField( pos + rray*aoRange )/(aoRange*max(.01,dot(rray,normal))) ); // can be > 1.0
	specOcclusion = exp2( -2.0*pow(specOcclusion,2.0) ); // tweak the curve
	
	// prevent sparkles in heavily occluded areas
	specOcclusion *= occlusion;

	reflection *= specOcclusion; // could fire an additional ray for more accurate results
	
	// fresnel
	float fresnel = pow( 1.0+dot(normal,ray), 5.0 );
	fresnel = lerp( .02, 1.0, fresnel );
	
	float3 result = float(0);

	
	// Combine all shading stages
	// comment these out to toggle various parts of the effect
	light += ambient;

//	light = mix( light, subsurface, .5 );
	
	result = light*albedo;

//	result = mix( result, reflection, fresnel );
	
	result += specular;

	return result;
}


// Isosurface Renderer
#ifdef FAST
int traceLimit=40;
float traceSize=.005;
#else
int traceLimit=100;
float traceSize=.001;//before *t: .002;
#endif	



float Trace( float3 pos, float3 ray, float traceStart, float traceEnd )
{
	float t = traceStart;
	float h;
	for( int i=0; i < traceLimit; i++ )
	{
		h = DistanceField( pos+t*ray );
		if ( h < traceSize*t || t > traceEnd )
			break;
		t = t+h;
	}
	
	if ( t > traceEnd )//|| h > .001 )
		return 0.0;
	
	return t;
}

float TraceMin( float3 pos, float3 ray, float traceStart, float traceEnd )
{
	float Min = traceEnd;
	float t = traceStart;
	float h;
	for( int i=0; i < traceLimit; i++ )
	{
		h = DistanceField( pos+t*ray );
		if ( h < .001 || t > traceEnd )
			break;
		Min = min(h,Min);
		t = t+max(h,.1);
	}
	
	if ( h < .001 )
		return 0.0;
	
	return Min;
}

float3 Normal( float3 pos, float3 ray, float t )
{
	// in theory we should be able to get a good gradient using just 4 points

	float pitch = .2 * t / R.x;
#ifdef FAST
	// don't sample smaller than the interpolation errors in Noise()
	pitch = max( pitch, .005 );
#endif
	
	float2 d = float2(-1,1) * pitch;

	float3 p0 = pos+d.xxx; // tetrahedral offsets
	float3 p1 = pos+d.xyy;
	float3 p2 = pos+d.yxy;
	float3 p3 = pos+d.yyx;
	
	float f0 = DistanceField(p0);
	float f1 = DistanceField(p1);
	float f2 = DistanceField(p2);
	float f3 = DistanceField(p3);
	
	float3 grad = p0*f0+p1*f1+p2*f2+p3*f3 - pos*(f0+f1+f2+f3);
	
	// prevent normals pointing away from camera (caused by precision errors)
	float gdr = dot ( grad, ray );
	grad -= max(.0,gdr)*ray;
	
	return normalize(grad);
}


// Camera

float3 Ray( float zoom, float4 fragCoord )
{
	return float3( fragCoord.xy-R.xy*.5, R.x*zoom );
}

float3 Rotate( inout float3 v, float2 a )
{
	float4 cs = float4( cos(a.x), sin(a.x), cos(a.y), sin(a.y) );
	
	v.yz = v.yz*cs.x+v.zy*cs.y*float2(-1,1);
	v.xz = v.xz*cs.z+v.zx*cs.w*float2(1,-1);
	
	float3 p;
	p.xz = float2( -cs.w, -cs.z )*cs.x;
	p.y = cs.y;
	
	return p;
}


// Camera Effects

void BarrelDistortion( inout float3 ray, float degree )
{
	// would love to get some disperson on this, but that means more rays
	ray.z /= degree;
	ray.z = ( ray.z*ray.z - dot(ray.xy,ray.xy) ); // fisheye
	ray.z = degree*sqrt(ray.z);
}

float3 LensFlare( float3 ray, float3 lightCol, float3 light, float lightVisible, float sky, float4 fragCoord )
{
	float2 dirtuv = fragCoord.xy/R.x;
	
	float dirt = 1.0-tex1.Sample( s0, dirtuv ).r;
	
	float l = (dot(light,ray)*.5+.5);
	
	return (
			((pow(abs(l),30.0)+.05)*dirt*.1
			+ 1.0*pow(abs(l),200.0))*lightVisible + sky*1.0*pow(abs(l),5000.0)
		   )*lightCol
		   + 5.0*pow(smoothstep(.9999,1.0,l),20.0) * lightVisible * normalize(lightCol);
}


float SmoothMax( float a, float b, float smoothing )
{
	return a-sqrt(smoothing*smoothing+pow(max(.0,a-b),2.0));
}

struct VS_IN
{
	float4 PosO:POSITION;
	float4 TexCd:TEXCOORD0;
};

struct vs2ps
{
    float4 PosWVP:SV_POSITION;
    float4 TexCd:TEXCOORD0;
};

vs2ps VS(VS_IN input)
{
    vs2ps output;
    output.PosWVP  = mul(input.PosO,tW); //mul(input.PosO,mul(tW,tVP));
    output.TexCd = input.TexCd;
    return output;
}

float4 PS(vs2ps In) : SV_Target
{
  
  float3 ray = Ray( .7, In.PosWVP );
	
	BarrelDistortion( ray, .5 );
	
	ray = normalize(ray);
	float3 localRay = ray;

    float2 rot = float2(.2,0.0);
    
	float T = time*.13;
	float3 mouse = float3(.2,.5,0);
	if ( mouse.z > .0 )
		rot += float2(1.0,-6.3) * (float(.5)-mouse.yx/R.yx);
    else
        rot -= float2(0,1)*T;
		
    float dist = 7.0;// * (-sin(iGlobalTime/10.0)+1.0);
	float3 pos = dist*Rotate( ray, rot );
	//pos += float3(0,.3,0) + T*float3(0,0,-1);
	
	float3 col;

	float3 lightDir = normalize(float3(3,1,-2));
	
	float3 lightCol = float3(1.1,1,.9)*1.0;
	
	// can adjust these according to the scene, even per-pixel to a bounding volume
	float near = .0;
	float far = 40.0;
	
	float t = Trace( pos, ray, near, far );
	if ( t > .0 )
	{
		float3 p = pos + ray*t;
		
		// shadow test
		float s = 0.0;
		//s = TraceMin( p, lightDir, .05, far );
        s = Trace( p, lightDir, .01*t, far );
		
		float3 n = Normal(p, ray, t);
		col = Shade( p, ray, n,  lightCol,
					//smoothstep( .0, .01, s ),
                    step( s, .001 ),
                    t );
		
		// fog
		float f = 200.0;
		col = lerp( float(.8), col, exp2(-t*float3(.4,.6,1.0)/f) );
	}
	else
	{
		col = Sky( ray );
	}
	
	// lens flare
	float s = TraceMin( pos, lightDir, .5, 40.0 );
	col += LensFlare( ray, lightCol, lightDir, smoothstep(.01,.1,s), step(t,.0),In.TexCd );
	
	// vignetting:
	col *= smoothstep( 1.0, .0, dot(localRay.xy,localRay.xy) );
	
	// compress bright colours, ( because bloom vanishes in vignette )
	float3 c = (col-1.0);
	c = sqrt(c*c+.05); // soft abs
	col = lerp(col,1.0-c,.48); // .5 = never saturate, .0 = linear
	
	// grain
	float2 grainuv = In.TexCd.xy + floor(time*60.0)*float2(37,41);
	float2 filmNoise = tex0.Sample( s0, .5*grainuv/R.xy ).rb;
	col *= lerp( float(1), lerp(float3(1,.5,0),float3(0,.5,1),filmNoise.x), .1*filmNoise.y );
	
	// compress bright colours
	float l = max(col.x,max(col.y,col.z));//dot(col,normalize(float3(2,4,1)));
	l = max(l,.01); // prevent div by zero, darker colours will have no curve
	float l2 = SmoothMax(l,1.0,.01);
	col *= l2/l;
	
	return float4(pow(abs(col),float(1.0/2.2)),1);
}


technique10 _4DGridSlice
{
	pass P0
	{
		SetVertexShader(CompileShader(vs_4_0,VS()));
		SetPixelShader(CompileShader(ps_4_0,PS()));
	
	}
}



