/*
	Bumped Sinusoidal Warp
	----------------------

	Sinusoidal planar deformation, or the 2D sine warp effect to people 
	like me. The effect has been around for years, and there are
	countless examples on the net. IQ's "Sculpture III" is basically a 
	much more sophisticated, spherical variation.

    This particular version was modified from Fabrice's "Plop 2," which in 
	turn was a simplified version of Fantomas's "Plop." I simply reduced 
	the frequency and iteration count in order to make it less busy.

	I also threw in a texture, added point-lit bump mapping, speckles... 
	and that's pretty much it. As for why a metallic surface would be 
	defying	the laws of physics and moving like this is anyone's guess. :)

	By the way, I have a 3D version, similar to this, that I'll put up at 
	a later date.


	Related examples:

    Fantomas - Plop
    https://www.shadertoy.com/view/ltSSDV

    Fabrice - Plop 2
    https://www.shadertoy.com/view/MlSSDV

	IQ - Sculpture III (loosely related)
	https://www.shadertoy.com/view/XtjSDK

	Shane - Lit Sine Warp (far less code)
	https://www.shadertoy.com/view/Ml2XDV

*/


float2 R:TARGETSIZE;
Texture2D tex0 <string uiname="Texture";>;
SamplerState s0:IMMUTABLE
{
    Filter = MIN_MAG_MIP_LINEAR;
    AddressU = wrap;
    AddressV = wrap;
};

cbuffer cbPerDraw:register( b0 )
{
float4x4 tVP:VIEWPROJECTION;
float4x4 tW:WORLD;
float time;
};


// Warp function. Variations have been around for years. This is
// almost the same as Fabrice's version:
// Fabrice - Plop 2
// https://www.shadertoy.com/view/MlSSDV
float2 W(float2 p)
{
    
    p = (p+3.)*4.;

    float t = time/2.;

    // Layered, sinusoidal feedback, with time component.
    for (int i=0; i<3; i++){
        p += cos( p.yx*3. + float2(t,1.57))/3.;
        p += sin( p.yx + t + float2(1.57,0))/2.;
        p *= 1.3;
    }

    // A bit of jitter to counter the high frequency sections.
    p +=  frac(sin(p+float2(13, 7))*5e5)*.03-.015;

    return fmod(p,2.)-1.; // Range: [vec2(-1), vec2(1)]
    
}

// Bump mapping function. Put whatever you want here. In this case, 
// we're returning the length of the sinusoidal warp function.
float bumpFunc(float2 p)
{

	return length(W(p))*.7071; // Range: [0, 1]

}

/*
// Standard ray-plane intersection.
float3 rayPlane(float3 p, float3 o, float3 n, float3 rd) 
{
    
    float dn = dot(rd, n);

    float s = 1e8;
    
    if (abs(dn) > 0.0001) {
        s = dot(p-o, n) / dn;
        s += float(s < 0.0) * 1e8;
    }
    
    return o + s*rd;
}
*/

//vec3 smoothFract(vec3 x){ x = fract(x); return min(x, x*(1.-x)*12.); }


struct VS_IN
{
	float4 PosO:POSITION;
	float4 TexCd:TEXCOORD0;
};

struct vs2ps
{
    float4 PosWVP:SV_POSITION;
    float4 TexCd:TEXCOORD0;
};

vs2ps VS(VS_IN input)
{
    vs2ps output;
    output.PosWVP = mul(input.PosO,tW);
    output.TexCd = input.TexCd;
    return output;
}

float4 PS(vs2ps In) : SV_Target
{
    // Screen coordinates.
	float2 uv = 1-In.TexCd ;
    uv.x *=  R.x / R.y;
    
    // PLANE ROTATION
    //
    // Rotating the canvas back and forth. I don't feel it adds value, in this case,
    // but feel free to uncomment it.
    //float th = sin(iGlobalTime*0.1)*sin(iGlobalTime*0.12)*4.;
    //float cs = cos(th), si = sin(th);
    //uv *= mat2(cs, -si, si, cs);
  

    // VECTOR SETUP - surface postion, ray origin, unit direction vector, and light postion.
    //
    // Setup: I find 2D bump mapping more intuitive to pretend I'm raytracing, then lighting a bump mapped plane 
    // situated at the origin. Others may disagree. :)  
    float3 sp = float3(uv, 0); // Surface posion. Hit point, if you prefer. Essentially, a screen at the origin.
    float3 rd = normalize(float3(uv, 1.)); // Unit direction vector. From the origin to the screen plane.
    float3 lp = float3(cos(time)*0.5, sin(time)*0.2, -1.); // Light position - Back from the screen.
	float3 sn = float3(0., 0., -1); // Plane normal. Z pointing toward the viewer.
 
     
/*
	// I deliberately left this block in to show that the above is a simplified version
	// of a raytraced plane. The "rayPlane" equation is commented out above.
	vec3 rd = normalize(vec3(uv, 1.));
	vec3 ro = vec3(0., 0., -1);

	// Plane normal.
	vec3 sn = normalize(vec3(cos(iGlobalTime)*0.25, sin(iGlobalTime)*0.25, -1));
    //vec3 sn = normalize(vec3(0., 0., -1));
	
	vec3 sp = rayPlane(vec3(0., 0., 0.), ro, sn, rd);
    vec3 lp = vec3(cos(iGlobalTime)*0.5, sin(iGlobalTime)*0.25, -1.); 
*/    
    
    
    // BUMP MAPPING - PERTURBING THE NORMAL
    //
    // Setting up the bump mapping variables. Normally, you'd amalgamate a lot of the following,
    // and roll it into a single function, but I wanted to show the workings.
    //
    // f - Function value
    // fx - Change in "f" in in the X-direction.
    // fy - Change in "f" in in the Y-direction.
    float2 eps = float2(4./R.y, 0.);
    
    float f = bumpFunc(sp.xy); // Sample value multiplied by the amplitude.
    float fx = bumpFunc(sp.xy-eps.xy); // Same for the nearby sample in the X-direction.
    float fy = bumpFunc(sp.xy-eps.yx); // Same for the nearby sample in the Y-direction.
   
 	// Controls how much the bump is accentuated.
	const float bumpFactor = 0.05;
    
    // Using the above to determine the dx and dy function gradients.
    fx = (fx-f)/eps.x; // Change in X
    fy = (fy-f)/eps.x; // Change in Y.
    // Using the gradient vector, "vec3(fx, fy, 0)," to perturb the XY plane normal ",vec3(0, 0, -1)."
    // By the way, there's a redundant step I'm skipping in this particular case, on account of the 
    // normal only having a Z-component. Normally, though, you'd need the commented stuff below.
    //vec3 grad = vec3(fx, fy, 0);
    //grad -= sn*dot(sn, grad);
    //sn = normalize( sn + grad*bumpFactor ); 
    sn = normalize( sn + float3(fx, fy, 0)*bumpFactor );           
   
    
    // LIGHTING
    //
	// Determine the light direction vector, calculate its distance, then normalize it.
	float3 ld = lp - sp;
	float lDist = max(length(ld), 0.001);
	ld /= lDist;

    // Light attenuation.    
    float atten = min(1./(0.25 + lDist*0.5 + lDist*lDist*0.05), 1.);
	//float atten = min(1./(lDist*lDist*1.), 1.);
    
    // Using the bump function, "f," to darken the crevices. Completely optional, but I
    // find it gives extra depth.
    atten *= f*.85 + .15; // Or... f*f*.5 + .5; //  pow(f, .75); // etc.

	

	// Diffuse value.
	float diff = max(dot(sn, ld), 0.);  
    // Enhancing the diffuse value a bit. Made up.
    diff = pow(diff, 2.)*0.66 + pow(diff, 4.)*0.34; 
    // Specular highlighting.
    float spec = pow(max(dot( reflect(-ld, sn), -rd), 0.), 8.); 

	
    // TEXTURE COLOR
    //
	// Combining the surface postion with a fraction of the warped surface position to index 
    // into the texture. The result is a slightly warped texture, as a opposed to a completely 
    // warped one. By the way, the warp function is called above in the "bumpFunc" function,
    // so it's kind of wasteful doing it again here, but the function is kind of cheap, and
    // it's more readable this way.
    float3 texCol = tex0.Sample(s0, sp.xy + W(sp.xy)/8.).xyz;
    //
    /*
    // Textureless. Simple and elegant... so it clearly didn't come from me. Thanks Fabrice. :)
    vec3 texCol = smoothFract( W(sp.xy).xyy )*.1 + .4;
	*/
    
    // FINAL COLOR
    // Using the values above to produce the final color.   
    float3 col = (texCol * (diff*float3(1, .97, .92)*1.3 + 0.5) + float3(1., 0.6, .2)*spec*1.3)*atten;


    // Done. 
	return float4(min(col, 1.), 1.);

}


technique10 BumpedSinusoidalWarp
{
	pass P0
	{
		SetVertexShader(CompileShader(vs_4_0,VS()));
		SetPixelShader(CompileShader(ps_4_0,PS()));
	
	}
}



