#region usings
using System;
using System.ComponentModel.Composition;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;

using VVVV.Core.Logging;
#endregion usings

namespace VVVV.Nodes
{
	#region PluginInfo
	[PluginInfo(Name = "uDrawHID", Category = "Devices", Help = "Decodes HID Messages from uDraw PS3 controller", Tags = "")]
	#endregion PluginInfo
	public class uDrawHIDNode : IPluginEvaluate
	{
		#region fields & pins
		[Input("HID Input", DefaultString = "")]
		ISpread<string> FInput;

		[Output("Position")]
		ISpread<Vector2D> FOutputPosition;

		[Output("Touch Type")]
		ISpread<int> FOutputTouchType;

		[Output("Pen Pressure")]
		ISpread<float> FOutputPenPressure;

		[Output("MultiTouch Distance")]
		ISpread<float> FOutputMultiTouchDistance;

		//[Output("MultiTouch Angle")]
		//ISpread<float> FOutputMultiTouchAngle;

		[Output("DPad")]
		ISpread<int> FOutputDPad;

		[Output("Buttons")]
		ISpread<int> FOutputButtons;

		[Output("Additional Buttons")]
		ISpread<int> FOutputAddButtons;

		[Output("Accelerometer")]
		ISpread<Vector3D> FOutputAccelerometer;
				
		#endregion fields & pins

		byte[] input;

		
		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			if (FInput[0].Length == 28) // absolute bare minimum sanity check ^^
			{
				//En-/Decoding is essential ;)
				input = System.Text.Encoding.GetEncoding(28591).GetBytes(FInput[0]);			
				
				//Position
				int posX = input[16]*256 + input[18];
				int posY = input[17]*256 + input[19];
				
				if (posX <= 1920 || posY <= 1080)
					FOutputPosition[0] = new Vector2D((float)posX/1920 , (float)posY/1080);
				
				//Touch Type
				if (input[12] == 64)
				{
					FOutputTouchType[0] = 2;
			
				} else if (input[12] == 128) 
				{
					FOutputTouchType[0] = 1;
				} else if (input[12] >= 191) 
				{
					FOutputTouchType[0] = 3;
					FOutputMultiTouchDistance[0] = (float)input[13]/255;
					//FOutputMultiTouchAngle[0] = (float)(input[12]-191)/63;
				} else
				{
					FOutputTouchType[0] = 0;
				}
				
				//Pressure
				FOutputPenPressure[0] = ((float)input[14]-104)/151;
								
				//DPad
				if (input[3] == 15)
				{
					FOutputDPad[0] = 0;
				} else {
					FOutputDPad[0] = input[3]+1;
				}
				
				//Buttons
				FOutputButtons.SliceCount = 4;
				
				if ((input[1] & 0x01) > 0 )
				{
					FOutputButtons[0]=1;
				} else
				{
					FOutputButtons[0]=0;
				}
				
				if ((input[1] & 0x02) > 0 )
				{
					FOutputButtons[1]=1;
				} else
				{
					FOutputButtons[1]=0;
				}
				
				if ((input[1] & 0x04) > 0 )
				{
					FOutputButtons[2]=1;
				} else
				{
					FOutputButtons[2]=0;
				}
				
				if ((input[1] & 0x08) > 0 )
				{
					FOutputButtons[3]=1;
				} else
				{
					FOutputButtons[3]=0;
				}
				
				//Additional Buttons
				FOutputAddButtons.SliceCount = 3;
				
				if ((input[2] & 0x01) > 0 )
				{
					FOutputAddButtons[0]=1;
				} else
				{
					FOutputAddButtons[0]=0;
				}
				
				if ((input[2] & 0x02) > 0 )
				{
					FOutputAddButtons[1]=1;
				} else
				{
					FOutputAddButtons[1]=0;
				}
				
				if ((input[2] & 0x14) > 0 )
				{
					FOutputAddButtons[2]=1;
				} else
				{
					FOutputAddButtons[2]=0;
				}
				
				//Accelerometer				
				const ushort X_MIN = 0x1EA; const ushort X_MAX = 0x216;
				const ushort Y_MIN = 0x1EA; const ushort Y_MAX = 0x216;
				const ushort Z_MIN = 0x1EC; const ushort Z_MAX = 0x218;
				
				float x = (input[20] | (input[21] << 8));
				if (x < X_MIN) x = X_MIN; if (x > X_MAX) x = X_MAX;
				float y = (input[22] | (input[23] << 8));
				if (y < Y_MIN) y = Y_MIN; if (y > Y_MAX) y = Y_MAX;
				float z = (input[24] | (input[25] << 8));
				if (z < Z_MIN) z = Z_MIN; if (z > Z_MAX) z = Z_MAX;

				FOutputAccelerometer[0] = new Vector3D
				(
					(float)((x - X_MIN) / (X_MAX - X_MIN)),
					(float)((y - Y_MIN) / (Y_MAX - Y_MIN)),
					(float)((z - Z_MIN) / (Z_MAX - Z_MIN))
				);
			}		
		}
		
	}
}
