#region usings
using System;
using System.ComponentModel.Composition;
using System.Collections.Generic;
using System.Linq;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;
using VVVV.Utils.Streams;

using VVVV.Core.Logging;
#endregion usings

namespace VVVV.Nodes
{
	public enum MappingEnum
	{
		Float,
		Wrap,
		Mirror,
		Clamp
	}
	
	public abstract class AdvancedMapNode : IPluginEvaluate
	{
		[Input("Mapping", DefaultEnumEntry = "Float", Order = 6 )]
		public IDiffSpread<MappingEnum> FMapping;
		
		[Output("Output")]
		protected ISpread<double> FOutput;
		
		
		[Import()]
		public ILogger FLogger;
		
		public virtual void Evaluate(int SpreadMax) {}
		
		protected double findRatio(double input, double min, double max, MappingEnum mapping) {
			double range = max - min;
			
			if (range == 0) return 0; // same behaviour was marked as BUG on the old delphi node. seems still a viable resolve though
			
			if (mapping == MappingEnum.Float) {
			}
			else if (mapping == MappingEnum.Clamp) {
				if (input < min) input = min;
				if (input > max) input = max;
			}
			else if (mapping == MappingEnum.Wrap) {
				double distance = (input - min);
				int rangeCount = (int)Math.Floor(distance / range);
				input -= range * (rangeCount);
			}
			else if (mapping == MappingEnum.Mirror) {
				var input2 = input;
				
				double distance = 0;
				if (input < min)  distance = (min - input);
				if (input >= max) distance = (max - input);
				
				// 				mirroring twice is the same as wrapping twice
				int rangeCount = (int)Math.Floor(distance / range);
				rangeCount = (rangeCount & 1)==1?rangeCount-1:rangeCount; // if uneven, make it even.
				input += range * (rangeCount);
				
				//				make sure its well inside
				if (input < min) input = 2*min - input;
				if (input > max) input = 2*max - input;
			}
			
			return (input-min)/range;
		}
	}
	
	
	#region PluginInfo
	[PluginInfo(Name = "Map", Category = "Value", Version = "Advanced", Help = "Maps the value in the given range to a proportional value in the given output range", Tags = "velcrome")]
	#endregion PluginInfo
	public class AdvancedValueMapNode : AdvancedMapNode
	{
		#region fields & pins
		[Input("Input", DefaultValue = 0.5, Order = 0)]
		public ISpread<double> FInput;
		
		[Input("Input Binsize", DefaultValue = 1, Order = 1, Visibility = PinVisibility.Hidden, MinValue=1)]
		public ISpread<int> FBinsize;
		
		[Input("Source Minimum", DefaultValue = 0.0, Order = 2)]
		public ISpread<double> FInMin;
		
		[Input("Source Maximum", DefaultValue = 1.0, Order = 3)]
		public ISpread<double> FInMax;
		
		[Input("Destination Minimum", DefaultValue = 0.0, Order = 4)]
		public ISpread<double> FOutMin;
		
		[Input("Destination Maximum", DefaultValue = 1.0, Order = 5)]
		public ISpread<double> FOutMax;
		
		
		#endregion fields & pins
		
		override public void Evaluate(int SpreadMax)
		{
			SpreadMax = FInput.SliceCount;
			FOutput.SliceCount = SpreadMax;
			
			double[] input = FInput.Stream.Buffer;
			int[] binsize = FBinsize.Stream.Buffer;
			
			double[] minI = FInMin.Stream.Buffer;
			double[] maxI = FInMax.Stream.Buffer;
			MappingEnum[] map = FMapping.Stream.Buffer;
			
			double[] minO = FOutMin.Stream.Buffer;
			double[] maxO = FOutMax.Stream.Buffer;
			
			double[] output = FOutput.Stream.Buffer;
			
			int binCounter = 0;
			int iBin = 0;
			
			int iMinL = 0;int iMaxL = 0;
			int oMinL = 0;int oMaxL = 0;
			int mapL = 0;
			
			for ( int index = 0 ; index < SpreadMax; index++) {
				// resetting counters is a lot less costly than a % operation
				if (binsize[iBin] == binCounter) {
					iMinL++;
					iMaxL++;
					oMinL++;
					oMaxL++;
					mapL++;
					binCounter = 0;
					
					if (iMinL >= minI.Length) iMinL=0;
					if (iMaxL >= maxI.Length) iMaxL=0;
					if (oMinL >= minO.Length) oMinL=0;
					if (oMaxL >= maxO.Length) oMaxL=0;
					if (mapL >= map.Length) mapL=0;
				}
				
				double ratio = findRatio(input[index], minI[iMinL], maxI[iMaxL], map[mapL]);
				FOutput[index] = VMath.Lerp(minO[oMinL], maxO[oMaxL], ratio);
			}
		}
	}
	
	
	
	
	#region PluginInfo
	[PluginInfo(Name = "MapRange", Category = "Value", Version = "Advanced", Help = "Maps the value in the given range to a proportional value in the given output range", Tags = "velcrome")]
	#endregion PluginInfo
	public class AdvancedValueMapRangeNode : AdvancedMapNode
	{
		#region fields & pins
		[Input("Input", DefaultValue = 0.5, Order = 0)]
		public ISpread<double> FInput;
		
		[Input("Input Binsize", DefaultValue = 1, Order = 1, Visibility = PinVisibility.Hidden, MinValue=1)]
		public ISpread<int> FBinsize;
		
		[Input("Source Center", DefaultValue = 0.5, Order = 2)]
		public ISpread<double> FInCenter;
		
		[Input("Source Width", DefaultValue = 1.0, Order = 3)]
		public ISpread<double> FInWidth;
		
		[Input("Destination Center", DefaultValue = 0.5, Order = 4)]
		public ISpread<double> FOutCenter;
		
		[Input("Destination Width", DefaultValue = 1.0, Order = 5)]
		public ISpread<double> FOutWidth;
		
		#endregion fields & pins
		
		//called when data for any output pin is requested
		override public void Evaluate(int SpreadMax)
		{
			SpreadMax = FInput.SliceCount;
			FOutput.SliceCount = SpreadMax;
			
			double[] input = FInput.Stream.Buffer;
			int[] binsize = FBinsize.Stream.Buffer;
			
			double[] cenI = FInCenter.Stream.Buffer;
			double[] widthI = FInWidth.Stream.Buffer;
			MappingEnum[] map = FMapping.Stream.Buffer;
			
			double[] cenO = FOutCenter.Stream.Buffer;
			double[] widthO = FOutWidth.Stream.Buffer;
			
			double[] output = FOutput.Stream.Buffer;
			
			int binCounter = 0;
			int iBin = 0;
			
			int iCenL = 0;int iWidthL = 0;
			int oCenL = 0;int oWidthL = 0;
			int mapL = 0;
			
			double halfWidth;
			for ( int index = 0 ; index < SpreadMax; index++) {
				// resetting counters is a lot less costly than a % operation
				if (binsize[iBin] == binCounter) {
					iCenL++;
					iWidthL++;
					oCenL++;
					oWidthL++;
					mapL++;
					binCounter = 0;
					
					if (iCenL >= cenI.Length) iCenL=0;
					if (iWidthL >= widthI.Length) iWidthL=0;
					if (oCenL >= cenO.Length) oCenL=0;
					if (oWidthL >= widthO.Length) oWidthL=0;
					if (mapL >= map.Length) mapL=0;
				}
				
				halfWidth =  widthI[iWidthL] / 2;
				double ratio = findRatio(input[index], cenI[iCenL] - halfWidth, cenI[iCenL] + halfWidth, map[mapL]);
				
				halfWidth =  widthO[oWidthL] / 2;
				FOutput[index] = VMath.Lerp(cenO[oCenL]-halfWidth, cenO[oCenL] + halfWidth, ratio);
			}
		}
	}
}
