#region usings
using System;
using System.ComponentModel.Composition;
using System.Collections.Generic;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;

using VVVV.Core.Logging;

using QuickGraph;
using QuickGraph.Algorithms;
#endregion usings

namespace VVVV.Graph
{
	//edge type
	public struct EdgeTag<TEdge>
	{
		public TEdge Tag;
		public double Weight;
		
		public EdgeTag(TEdge tag, double weight)
		{
			Tag = tag;
			Weight = weight;
		}
	}
	
	#region create the graph
	public class GraphCreateGraphNode<TVertex, TEdge>: IPluginEvaluate
	{
		#region fields & pins
		[Input("From Vertex", DefaultString = "A")]
		IDiffSpread<TVertex> FFrom;
		
		[Input("To Vertex", DefaultString = "B")]
		IDiffSpread<TVertex> FTo;
		
		[Input("Edge", DefaultString = "A to B")]
		IDiffSpread<TEdge> FEdge;
		
		[Input("Edge Weight", DefaultValue = 1)]
		IDiffSpread<double> FWeight;
		
		[Input("Is Directed", DefaultValue = 1, IsSingle = true)]
		IDiffSpread<bool> FIsDirected;
		
		[Input("Allow Parallel Edges", IsSingle = true)]
		IDiffSpread<bool> FAllowParallel;

		[Output("Output")]
		ISpread<IGraph<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>> FOutput;

		[Import()]
		ILogger FLogger;
		#endregion fields & pins

		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			FOutput.SliceCount = 1;
			
			if(FFrom.IsChanged || FTo.IsChanged || FEdge.IsChanged || FWeight.IsChanged || FIsDirected.IsChanged || FAllowParallel.IsChanged)
			{
				IMutableVertexAndEdgeSet<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>> g;
				
				if(FIsDirected[0])
					g = new AdjacencyGraph<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>(FAllowParallel[0]);
				else
					g = new UndirectedGraph<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>(FAllowParallel[0]);
				
				for(int i=0; i<SpreadMax; i++)
				{
					var tag = new EdgeTag<TEdge>(FEdge[i], FWeight[i]);
					var e = new STaggedEdge<TVertex, EdgeTag<TEdge>>(FFrom[i], FTo[i], tag);
					g.AddVerticesAndEdge(e);
				}
				FOutput[0] = g;
			}
			//FLogger.Log(LogType.Debug, "Logging to Renderer (TTY)");
		}
	}
	#endregion create the graph
	
	#region get edges
	public class GraphGetEdges<TVertex, TEdge>: IPluginEvaluate
	{
		#region fields & pins
		[Input("Input")]
		ISpread<IGraph<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>> FGraph;
		
		[Output("From")]
		ISpread<TVertex> FFrom;
		
		[Output("To")]
		ISpread<TVertex> FTo;
		
		[Output("Edge")]
		ISpread<TEdge> FEdge;
		
		[Output("Weight")]
		ISpread<double> FWeight;

		[Import()]
		ILogger FLogger;
		#endregion fields & pins

		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			
			if (FGraph[0] != null)
			{
				var g = FGraph[0] as IEdgeSet<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>;
				SpreadMax = g.EdgeCount;
				FFrom.SliceCount = SpreadMax;
				FTo.SliceCount = SpreadMax;
				FEdge.SliceCount = SpreadMax;
				FWeight.SliceCount = SpreadMax; 
				
				int i = 0;
				foreach(STaggedEdge<TVertex, EdgeTag<TEdge>> e in g.Edges)
				{
					FFrom[i] = e.Source;
					FTo[i] = e.Target;
					FEdge[i] = e.Tag.Tag;
					FWeight[i] = e.Tag.Weight;
					i++;
				}	

			}
			//FLogger.Log(LogType.Debug, "Logging to Renderer (TTY)");
		}
	}
	#endregion get edges
	
	#region shortest path
	
	public enum ShortestPathAlgorithm
	{
		Dijkstra,
		AStar,
		BellmanFord,
		Dag
	}
	
	public class GraphShortestPath<TVertex, TEdge>: IPluginEvaluate where TVertex : IComparable
	{
		#region fields & pins
		[Input("Input")]
		ISpread<IGraph<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>> FGraph;
		
		[Input("From")]
		ISpread<TVertex> FFrom;
		
		[Input("To")]
		ISpread<TVertex> FTo;
		
		[Input("Algorithm for Directed Graph")]
		ISpread<ShortestPathAlgorithm> FAlgorithm;
		
		[Output("Path Vertices")]
		ISpread<ISpread<TVertex>> FVertexPath;
		
		[Output("Path Edges")]
		ISpread<ISpread<TEdge>> FEdgePath;
		
		[Output("Path Weights")]
		ISpread<ISpread<double>> FWeightPath;
		
		[Output("Error")]
		ISpread<string> FError;
		
		[Import()]
		ILogger FLogger;
		#endregion fields & pins
		
		//called when data for any output pin is requested
		public void Evaluate(int SpreadMax)
		{
			FVertexPath.SliceCount = SpreadMax;
			FEdgePath.SliceCount = SpreadMax;
			FWeightPath.SliceCount = SpreadMax;
			FError.SliceCount = SpreadMax;
			
			if (FGraph[0] != null)
			{
				var g = FGraph[0];
				
				Func<STaggedEdge<TVertex, EdgeTag<TEdge>>, double> edgeCost = e => e.Tag.Weight;
				
				for(int i=0; i<SpreadMax; i++)
				{
					var root = FFrom[i];
					TryFunc<TVertex, IEnumerable<STaggedEdge<TVertex, EdgeTag<TEdge>>>> tryGetPaths;
					
					// compute shortest paths
					if(g.IsDirected)
					{
						var gr = g as IVertexAndEdgeListGraph<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>;
						if(!gr.ContainsVertex(root))
						{
							FError[i] = "From vertex not found: " + root.ToString();
							continue;
						}
						
						switch (FAlgorithm[0])
						{
							case ShortestPathAlgorithm.Dijkstra:
							tryGetPaths = gr.ShortestPathsDijkstra(edgeCost, root);
							break;
							
							case ShortestPathAlgorithm.AStar:
							Func<TVertex, double> vertexCost = v => 1;
							tryGetPaths = gr.ShortestPathsAStar(edgeCost, vertexCost, root);
							break;
							
							case ShortestPathAlgorithm.BellmanFord:
							tryGetPaths = gr.ShortestPathsBellmanFord(edgeCost, root);
							break;
							
							default:
							
							try
							{
								tryGetPaths = gr.ShortestPathsDag(edgeCost, root);
								
							} 
							catch (NonAcyclicGraphException e)
							{
								tryGetPaths = null;
								FError[i] = "This graph is not acyclic, the DAG path search algorithm can't be used with it.";
							}
							break;
						}
					}
					else
					{
						var gr = g as IUndirectedGraph<TVertex, STaggedEdge<TVertex, EdgeTag<TEdge>>>;
						if(!gr.ContainsVertex(root))
						{
							FError[i] = "From vertex not found: " + root.ToString();
							continue;
						}
						
						tryGetPaths = gr.ShortestPathsDijkstra(edgeCost, root);
					}
					
					var vp = FVertexPath[i];
					var ep = FEdgePath[i];
					var wp = FWeightPath[i];
					vp.SliceCount = 0;
					ep.SliceCount = 0;
					wp.SliceCount = 0;
					
					if (tryGetPaths == null) continue;
					
					// query path for given vertices
					TVertex target = FTo[i];
					IEnumerable<STaggedEdge<TVertex, EdgeTag<TEdge>>> path;
					if (tryGetPaths(target, out path))
					{
						vp.Add(root);
						
						//enumerate the path
						foreach (var e in path)
						{
							ep.Add(e.Tag.Tag);
							wp.Add(e.Tag.Weight);
							if(g.IsDirected) //no need to order the vertices
							{
								vp.Add(e.Target);
							}
							else //order vertices
							{
								if(!vp[vp.SliceCount - 1].Equals(e.Target))
								vp.Add(e.Target);
								else
								vp.Add(e.Source);
							}
						}
						
						FError[i] = "OK";
					}
					else
					{
						FError[i] = "Can't find path from: " + root.ToString() + " to: " + target.ToString();
					}
				}
			}
		}
	}
	
	#endregion shortest path
}